# coding=utf-8
import distutils.dir_util
import logging
import os
import shutil

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common import fs
from sandbox.common.utils import get_task_link
from sandbox.projects.common import binary_task
from sandbox.projects.metrika.utils.base_metrika_task.base_metrika_task import BaseMetrikaLargeTask
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.admins.cosmos.cosmos_runner.view import ViewModel
from sandbox.projects.metrika.admins.cosmos.utils import CosmosReport, NoTestsRunException, ReportNotGenerated, TestFailures, HazelcastParameters, ReportStartrekParameters, YavParameters, \
    CheckTestsResultParameters, ReportTtlParameters
from sandbox.projects.metrika.admins.cosmos.utils import PackDir
from sandbox.projects.metrika.admins.cosmos.utils.surefire_report import parse_surefire_reports
from sandbox.projects.metrika.utils import CommonParameters, render, custom_report_logger
from sandbox.projects.metrika.utils import parameters
from sandbox.projects.metrika.utils.base_metrika_task import with_parents
from sandbox.projects.metrika.utils.mixins import maven
from sandbox.sdk2 import paths, Path


@with_parents
class CosmosRunner(BaseMetrikaLargeTask, maven.BaseMavenMixin):
    """
    Выполнятель сьютов - пак
    """

    class Parameters(CommonParameters):
        container = parameters.LastCosmosContainerResource("Environment container resource", required=True)
        pack_id = sdk2.parameters.String('Идентификатор пака', required=True,
                                         description='Имя каталога, содержащего описание пака для запуска')
        override_tests = sdk2.parameters.Bool('Запустить указанные тесты', required=True, default=False,
                                              description='Будут запущены тесты не из пака, а только те, что здесь перечислены явно. Используется для перезапуска только упавших.')
        with override_tests.value[True]:
            tests = sdk2.parameters.List('Тесты', required=True,
                                         description='Описание запуска тестов, по одному на строку, как в описании пака. '
                                                     'Подробности в https://maven.apache.org/surefire/maven-surefire-plugin/examples/single-test.html')
        test_properties = sdk2.parameters.Dict('Test properties', default={}, required=False,
                                               description='Словарь дополнительных Java-properties для запуска тестов')
        artifacts_resource = sdk2.parameters.Resource('Артефакты тестов', required=True,
                                                      description='Ресурс, содержащий локальный maven-репозиторий с необходимыми для данного пака артефактами и каталог с описаниями паков')
        report_ttl_params = ReportTtlParameters()
        yav = YavParameters()
        reporting = ReportStartrekParameters()
        hz = HazelcastParameters()
        check_test_results = CheckTestsResultParameters()

        with sdk2.parameters.Output:
            with sdk2.parameters.Group('Результаты тестов') as output:
                report_resource = sdk2.parameters.Resource('Отчёты', required=True,
                                                           description='Ресурс, содержащий отчёты после прогона пака')
                test_results = sdk2.parameters.JSON('Результаты тестов', required=True,
                                                    description='Краткие результаты тестов')
                rerun_task = sdk2.parameters.Task('Задача на перезапуск упавших тестов', required=False,
                                                  description='Если есть упавшие тесты, то будет создана задача на перезапуск, её нужно будет вручную явно запустить.')

        _binary = binary_task.binary_release_parameters_list(stable=True)

    class Context(sdk2.Task.Context):
        name = None
        title = None

    def on_execute(self):
        os.environ.update({"VAULT_TOKEN": self.Parameters.yav_token.data()})
        with sdk2.helpers.ProgressMeter("Download bundle"):
            self._download_bundle()
            self.Context.name = self._pack().name
            self.Context.title = self._pack().title
            self.Context.save()

        with sdk2.helpers.ProgressMeter("Run tests for artifacts"):
            self._run_artifacts()

        with sdk2.helpers.ProgressMeter("Build report"):
            self._build_report()

        with sdk2.helpers.ProgressMeter("Analyze result"):
            if self.Parameters.test_results["total"] == 0:
                raise NoTestsRunException()
            elif self.Parameters.test_results["failed"] > 0:
                self.Parameters.rerun_task = self._create_rerun_task()
                self.set_info('<a href="{link}">#{id}</a> {description} (перезапуск упавших тестов)'.format(
                    link=get_task_link(self.Parameters.rerun_task.id),
                    id=self.Parameters.rerun_task.id,
                    description=self.Parameters.rerun_task.Parameters.description.encode('utf-8', errors='replace')),
                    do_escape=False)
        if self.Parameters.report_startrek:
            with sdk2.helpers.ProgressMeter("Report to Startrek issue {}".format(self.Parameters.issue_key)):
                self._report_startrek()

        if self.Parameters.fail_task_on_test_failure:
            if not self.is_all_tests_passed:
                raise TestFailures()

    def _report_path(self):
        return self.path('report')

    def _pack(self):
        return PackDir(Path(self.wd('packs', self.Parameters.pack_id)))

    def _get_maven_local_repo_path(self):
        return Path(self.wd("repository"))

    def _download_bundle(self):
        fs.copy_dir(sdk2.ResourceData(self.Parameters.artifacts_resource).path.as_posix(), self.wd())
        paths.add_write_permissions_for_path(self.wd())

    def _run_artifacts(self):
        # FIXME пока запускаем последовательно
        # у нас обычно в паке один артефакт
        # для противоположного случая - пак собран из нескольких артефактов - нужно будет сделать параллельный запуск maven'а
        # либо ещё запускать по одной строке из файла test по всем артефактам - в отдельном вызове maven'а - тоже параллельно
        for artifact in self._pack().artifacts:
            props = {
                'aero.pack.uuid': '0000',  # backward compatibility
                'aero.suite.uuid': '0000',  # backward compatibility
                'aero.suite.name': 'SuiteName0000',  # backward compatibility
                'hazelcast.group.name': self.Parameters.hz.group_name.data(),
                'hazelcast.group.password': self.Parameters.hz.group_password.data(),
                'test': ', '.join(self.Parameters.tests if self.Parameters.override_tests else artifact.test)
            }
            props.update(dict(self._pack().properties))
            props.update(self.Parameters.test_properties)
            self._execute_maven(['test', '--file', artifact.run_pom_path.as_posix()], properties=props)

    def _build_report(self):
        # соберём файлы от запусков
        for artifact in self._pack().artifacts:
            for target_item in [item.name for item in artifact.target().iterdir() if item.is_dir() and item.name in ["allure-results", "site", "surefire-reports"]]:
                distutils.dir_util.copy_tree(artifact.target(target_item).as_posix(), self._pack().target(target_item).as_posix())

        try:
            # junit файлы распарсим и получим результат, тут возможен невалидный xml и т.п.
            self.Parameters.test_results = parse_surefire_reports(self._pack().target("surefire-reports"))

            # выполним генерацию репорта если тут упадёт, то сохраним таки артефакт, что бы потом что-то как-то перегенировать
            # здесь не нужно вызывать COSMOS_REPORTER, т.к. это увеличивает накладные расходы
            self._execute_maven(['site', '--file', self._pack().report_pom_path.as_posix()])
        except:
            logging.warning("Ошибка в генерации отчёта, чего быть не должно.", exc_info=True)
            raise ReportNotGenerated()
        finally:
            # заархивируем в артефакт то, что удалось сделать - пригодится для дальнейших разбирательств
            shutil.move(self._pack().target().as_posix(), self._report_path().as_posix())
            report_resource = CosmosReport(self, self._pack().title, self._report_path(), ttl=self.Parameters.report_ttl,
                                           artifact_resource_id=self.Parameters.artifacts_resource.id,
                                           pack_id=self._pack().name)

            sdk2.ResourceData(report_resource).ready()

            self.Parameters.report_resource = report_resource

    def _create_rerun_task(self):
        params = dict(self.Parameters)
        del params[CosmosRunner.Parameters.report_resource.name]
        del params[CosmosRunner.Parameters.test_results.name]
        del params[CosmosRunner.Parameters.rerun_task.name]
        params.update({
            CosmosRunner.Parameters.owner.name: self.owner,
            CosmosRunner.Parameters.override_tests.name: True,
            CosmosRunner.Parameters.tests.name: self.Parameters.test_results["failed_suites"]
        })

        task = CosmosRunner(self, **params)
        if self.Parameters.binary_executor_release_type in {'custom', 'testing'}:
            task.Requirements.tasks_resource = self.Requirements.tasks_resource
            task.save()
        return task

    def _report_startrek(self):
        # этот код не должен ничего ломать
        try:
            issue = self.st_client.issues[self.Parameters.issue_key]
            issue.comments.create(text=self.get_wiki_comment())
        except Exception as e:
            self.set_info(e.message)
            logging.warning("Ошибка в отправке комментария в Startrek", exc_info=True)

    @sdk2.header()
    @custom_report_logger
    def header(self):
        return render("header.html.jinja2", {"view": ViewModel(self)})

    @sdk2.footer()
    @custom_report_logger
    def footer(self):
        return render("footer.html.jinja2", {"view": ViewModel(self)})

    @sdk2.report(utils.display("Перезапуски"))
    @custom_report_logger
    def reruns_report(self):
        return render("reruns.html.jinja2", {"view": ViewModel(self)})

    @sdk2.report(utils.display("wiki-комментарий"))
    @custom_report_logger
    def wiki_comment(self):
        return "<pre>" + self.get_wiki_comment() + "</pre>"

    def get_wiki_comment(self):
        return render("wiki.txt.jinja2", {"view": ViewModel(self)})

    @property
    def is_all_tests_passed(self):
        if self.Parameters.test_results:
            return int(self.Parameters.test_results["failed"]) == 0
        else:
            raise errors.TaskError("Результаты тестов не доступны")
