# coding=utf-8
import inspect

from sandbox import sdk2
from sandbox.common.errors import TaskFailure, TaskError
from sandbox.common.types.resource import RestartPolicy
from sandbox.projects.metrika.admins.cosmos.utils.packs import PackDir
from sandbox.sdk2 import Attributes


class CosmosArtifacts(sdk2.Resource):
    """
    Результат сборки тестов для их последующего запуска
    """
    executable = False
    auto_backup = False
    any_arch = True
    restart_policy = RestartPolicy.IGNORE
    share = False
    # версия артифактов, которая лежит в данном ресурсе
    artifact_version = Attributes.String("Artifact version", required=True, default=None)


class CosmosReport(sdk2.Resource):
    """
    Отчёты прогона всех тестов некоторого пака
    """
    executable = False
    auto_backup = True
    any_arch = True
    restart_policy = RestartPolicy.IGNORE
    share = False
    # id ресурсах с артефактами, использовавшимися для прогона тестов
    artifact_resource_id = Attributes.Integer("Artifact resource id", required=True)
    pack_id = Attributes.String("Pack name (id)", required=True)


class NoTestsRunException(TaskFailure):
    def get_task_info(self):
        return "Не было запущено ни одного теста"


class ReportNotGenerated(TaskError):
    def get_task_info(self):
        return "Отчёт не построен"


class TestFailures(TaskFailure):
    def get_task_info(self):
        return "Обнаружены проваленные тесты"


class HazelcastParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group("Hazelcast") as hz_block:
        group_name = sdk2.parameters.Vault("Hazelcast group name", required=True, default="METRIKA:HAZELCAST_GROUP_NAME",
                                           description="Секрет в котором хранится значение property hazelcast.group.name")
        group_password = sdk2.parameters.Vault("Hazelcast group password", required=True,
                                               default="METRIKA:HAZELCAST_GROUP_PASSWORD",
                                               description="Секрет в котором хранится значение property hazelcast.group.password")


class ReportStartrekParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group("Startrek") as st_block:
        report_startrek = sdk2.parameters.Bool('Оставить комментарий с результатами тестов', required=True, default=False)
        with report_startrek.value[True]:
            issue_key = sdk2.parameters.String('Issue Key', required=True,
                                               description='Ключ тикета в котором нужно оставить комментарий с результатами тестов')
            startrek_token = sdk2.parameters.Vault("OAuth-токен для доступа к Startrek", required=True, default="METRIKA:robot-metrika-test-tracker-token")


class YavParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group("Yandex Vault") as yav_block:
        yav_token = sdk2.parameters.Vault("OAuth-токен для доступа к секретнице", required=True,
                                          default="METRIKA:robot-metrika-test-yav")


class CheckTestsResultParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group("Анализ результатов тестов") as test_analyze_block:
        fail_task_on_test_failure = sdk2.parameters.Bool('Статус FAILURE, если есть упавшие сьюты', required=True,
                                                         default=False)


class ReportTtlParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group("Длительность хранения отчёта (TTL)") as report_ttl_block:
        report_ttl = sdk2.parameters.Integer("Длительность хранения отчёта, дн. (ttl)", required=True, default_value=30,
                                             description="Значение атрибута ttl ресурса с отчётом. Inf не поддерживается специально.")


class PackInfo(object):
    def __init__(self, state=None):
        self.state = state

    def init(self, pack_dir):
        assert self.state is None
        self.state = {}

        assert pack_dir is not None
        pack_dir_object = PackDir(pack_dir)

        self.state['pack_id'] = pack_dir_object.name
        self.state['title'] = pack_dir_object.title
        self.state['task_id'] = None

        return self

    @property
    def pack_id(self):
        return self.state[inspect.currentframe().f_code.co_name]

    @property
    def title(self):
        return self.state[inspect.currentframe().f_code.co_name]

    @property
    def task_id(self):
        return self.state[inspect.currentframe().f_code.co_name]

    @task_id.setter
    def task_id(self, value):
        self.state[inspect.currentframe().f_code.co_name] = value
