# coding=utf-8
import itertools
import logging

from sandbox import sdk2

logger = logging.getLogger('cosmos')

LOCAL_VERSION = '1.0-LOCAL'


class PacksDir(object):
    """
    Каталог с паками
    """

    def __init__(self, packs_dir):
        """
        :param packs_dir :type sdk2.Path
        """
        logger.debug("Packs dir: %s", packs_dir)
        self.packs_dir = packs_dir

    @property
    def packs(self):
        return [PackDir(pack_dir) for pack_dir in self.packs_dir.iterdir() if pack_dir.is_dir()]


class PackDir(object):
    """
    Каталог - пак
    """

    def __init__(self, pack_dir):
        """
        :param pack_dir :type sdk2.Path
        """
        logger.debug("Pack dir: %s", pack_dir)
        self.pack_dir = pack_dir

    @property
    def name(self):
        return self.pack_dir.name

    @property
    def properties(self):
        props = []
        for prop_file in self.pack_dir.glob('*.properties'):
            if prop_file.is_file():
                with prop_file.open(encoding='UTF-8', errors='replace') as f:
                    for line in f:
                        if not line.startswith('#'):
                            pair = line.strip().split('=', 1)
                            if len(pair) == 1:
                                pair.append('')
                            pair[1] = pair[1].encode('utf-8', errors='replace')
                            props.append(pair)

        return props

    @property
    def artifacts(self):
        artifacts = []
        for artifact_dir in self.pack_dir.iterdir():
            if artifact_dir.is_dir():
                artifacts.append(ArtifactDir(artifact_dir))
        return artifacts

    @property
    def title(self):
        title_path = self.pack_dir / "title"  # type: sdk2.Path
        if title_path.exists() and title_path.is_file():
            with title_path.open(encoding='UTF-8', errors='replace') as title_file:
                return title_file.readline().strip().encode('utf-8', errors='replace')
        else:
            return self.name

    @property
    def report_pom_path(self):
        return self.pack_dir / "report.pom.xml"

    def target(self, *path):
        return self.pack_dir.joinpath("target").joinpath(*path)


class ArtifactDir(object):
    """
    Каталог - описание артефакта
    """

    def __init__(self, artifact_dir):
        """
        :param artifact_dir :type sdk2.Path
        """
        logger.debug("Artifact dir: %s", artifact_dir)
        self.artifact_dir = artifact_dir

    @property
    def name(self):
        return self.artifact_dir.name

    @property
    def maven_version(self):
        """
        Если не указана версия, то дописывает локальную - LOCAL_VERSION
        :return:
        """
        artifact = {
            "group_id": None,
            "artifact_id": None,
            "version": None
        }
        artifact_file = self.artifact_dir / "artifact"  # type: sdk2.Path
        if artifact_file.exists() and artifact_file.is_file():
            with artifact_file.open(encoding='UTF-8', errors='replace') as f:
                artifact_line = f.readline().strip()
                split = artifact_line.split(":")
                if len(split) == 2:
                    artifact['group_id'] = split[0]
                    artifact['artifact_id'] = split[1]
                    artifact['version'] = LOCAL_VERSION
                elif len(split) == 3:
                    artifact['group_id'] = split[0]
                    artifact['artifact_id'] = split[1]
                    artifact['version'] = split[2]
                else:
                    raise Exception("Wrong artifact: " + artifact_line)

        return artifact

    @property
    def test(self):
        """
        :return: единый плоский список спецификации тестов - конкатенация списков из файлов по маске *.test
        """
        return list(itertools.chain(*self.test_chunks))

    @property
    def test_chunks(self):
        """
        :return: список списков тестов, как указано в файлах по маске *.test
        """
        chunks = []
        for tests_file in self.artifact_dir.glob('*.test'):
            if tests_file.is_file():
                with tests_file.open(encoding='UTF-8', errors='replace') as f:
                    chunks.append([line.strip().encode('utf-8', errors='replace') for line in f])

        return chunks

    @property
    def run_pom_path(self):
        return self.artifact_dir / "run.pom.xml"

    def target(self, *path):
        return self.artifact_dir.joinpath("target").joinpath(*path)
