from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.metrika.utils import CommonParameters, bstr
from sandbox.projects.metrika.utils.base_metrika_task import BaseMetrikaTask


class MetrikaDictionary(sdk2.Resource):
    share = True
    any_arch = True
    dict_name = sdk2.Attributes.String("Dictionary name", default_value="")


class MetrikaBuildDictionaryCommon(BaseMetrikaTask):
    config = None

    class Parameters(CommonParameters):
        description = 'Build metrika dictionary'
        kill_timeout = 600
        dict_name = sdk2.parameters.String("Dictinary name", default='OS')

        with sdk2.parameters.Group("Bishop"):
            bishop_program = sdk2.parameters.String("Bishop program name", default='metrika-build-dictionary')
            bishop_environment = sdk2.parameters.String("Bishop environment name", default='metrika.sandbox.admin.testing')

        with sdk2.parameters.Output:
            dict_resource = sdk2.parameters.Resource("Dictionary resource", resource_type=MetrikaDictionary)

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def on_execute(self):
        import metrika.pylib.log
        import metrika.pylib.config as lib_config
        import metrika.pylib.vault as mtv

        self.dict_file = self.path('wd', 'dicts', self.Parameters.dict_name + '.txt')
        self.dict_file.parent.mkdir(parents=True)

        metrika.pylib.log.init_logger('', stdout=True)
        token = sdk2.Vault.data(self.owner, "robot-metrika-admin-oauth")
        self.config = lib_config.get_yaml_config_from_bishop(
            program=self.Parameters.bishop_program,
            environment=self.Parameters.bishop_environment,
            token=token,
            vault_client=mtv.VaultClient(auth_type="oauth", oauth_token=token))

        self.make_dict()
        self.Parameters.dict_resource = MetrikaDictionary(self, self.Parameters.dict_name, self.dict_file.parent, dict_name=self.Parameters.dict_name)
        if 'bstr' in self.dict_config:
            self.bstr_push()
        if 's3' in self.dict_config:
            self.upload_to_s3()

    @property
    def dict_config(self):
        return self.config.dicts[self.Parameters.dict_name]

    def make_dict(self):
        """
        override in concrete tasks, write dictionary content to file by sdk2.path self.dict_file
        """
        raise NotImplementedError()

    def bstr_push(self):
        bstr.bstr_push(file=self.dict_file, **self.dict_config.bstr)

    def upload_to_s3(self):
        import boto3
        from boto3.s3.transfer import S3Transfer
        client = boto3.client('s3', **self.dict_config.s3.client)
        transfer = S3Transfer(client)
        transfer.upload_file(self.dict_file.as_posix(), self.dict_config.s3.bucket, self.Parameters.dict_name + '.txt')
