# coding=utf-8
from __future__ import unicode_literals

from collections import defaultdict
from datetime import date, datetime, timedelta
from time import mktime

from sandbox.common import rest
from sandbox.projects.common import binary_task
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.java.utils import metrika_java_helper
from sandbox.projects.metrika.utils import parameters as metrika_parameters, settings
from sandbox.projects.metrika.utils.base_metrika_task import with_parents
from sandbox.projects.metrika.utils.mixins.juggler_reporter import JugglerReporterMixin
from sandbox.projects.metrika.utils.pipeline.pipeline import PipelineBaseTask
from sandbox.sdk2 import parameters, yav


@with_parents
class MetrikaDutyTicket(PipelineBaseTask, JugglerReporterMixin):
    ESCALATIONS_TAGS = {
        'core': ['metrica-core'],
        'admins': ['metrica-infra', 'metrica-infra-sharded'],
        'api': ['metrica-api']
    }

    class Parameters(utils.CommonParameters):
        duration = parameters.Integer('Длительность дежурства в днях', required=True, default=7)

        with parameters.Group('Startrek') as startrek:
            summary_template = parameters.String('Шаблон названия тикета', required=True, default='Дежурство {duty_group} {date_from} - {date_to}')
            queue = metrika_parameters.RegexpString('Очередь', regexp=r'[A-Z]+', required=True)
            followers = parameters.List('Наблюдатели')
            components = parameters.List('Компоненты', value_type=parameters.Integer)
            ticket_description = parameters.String('Описание', multiline=True)
            st_tags = parameters.List('Теги')
            board = parameters.Integer('Доска')
            start_date = metrika_parameters.RegexpString('Дата начала дежурства', regexp=r'\d{4}-\d{2]-\d{2}', description='YYYY-MM-DD')
            group = parameters.String('Группа', required=True)

        schedulers = parameters.List('Шедулеры, в которые будет прописан ключ созданного тикета', required=False)

        _binary = binary_task.binary_release_parameters_list(stable=True)

    @property
    def juggler_host(self):
        return self.Parameters.group

    def create_stages(self):
        return [
            (self.ticket, 'Создание тикета'),
            (self.escalations, 'Статистика по дежурству')
        ]

    def ticket(self):
        from metrika.pylib import duty
        try:
            prefix = self.Parameters.summary_template[:self.Parameters.summary_template.index('{')]
        except ValueError:
            prefix = ''

        group_tag = 'duty_' + self.Parameters.group
        for previous_ticket in self.st_client.issues.find(filter=dict(queue=self.Parameters.queue, tags=group_tag), order='-createdAt'):
            if previous_ticket.summary.startswith(prefix):
                self.Context.previous_ticket_key = previous_ticket.key
                break
        else:
            previous_ticket = None

        duty_group = duty.DutyAPI(token=yav.Secret(settings.yav_uuid).data()['oauth-token']).get_duty_group(
            'metrika', self.Parameters.group
        )

        if self.Parameters.start_date:
            start_date = datetime.strptime(self.Parameters.start_date, '%Y-%m-%d').date()
        else:
            start_date = date.today()

        summary = self.Parameters.summary_template.format(
            duty_group=duty_group['name'],
            date_from=start_date,
            date_to=start_date + timedelta(days=self.Parameters.duration)
        )

        year, week, day = date.today().isocalendar()
        week_tag = '{}_week_{}'.format(year, week)
        search_parameters = {
            'queue': self.Parameters.queue,
            'tags': week_tag,
            'assignee': duty_group['duty'],
            'type': 'task'
        }
        if self.Parameters.components:
            search_parameters['components'] = self.Parameters.components
        create_parameters = search_parameters.copy()
        create_parameters.update({
            'tags': self.Parameters.st_tags + ['duty', group_tag, week_tag],
            'summary': summary,
            'description': self.Parameters.ticket_description,
            'followers': self.Parameters.followers,
            'boards': self.Parameters.board or None
        })
        issue = metrika_java_helper.MetrikaJavaHelper.find_or_create_issue(self.st_client, search_parameters, create_parameters, self.id)

        self.comment("Тикет: <a href='https://st.yandex-team.ru/{0}'>{0}</a>".format(issue.key))

        if previous_ticket and previous_ticket.key != issue.key:
            previous_ticket.links.create('relates', issue.key)

        if self.Parameters.schedulers:
            client = rest.Client()
            for scheduler in self.Parameters.schedulers:
                client.scheduler[scheduler].update({'task': {'custom_fields': [{'name': 'tracker_issue', 'value': issue.key}]}})

    def _get_escalations(self, filters, start_time, end_time):
        result = {}

        payload = {
            'filters': filters,
            'only_running': False,
            'page_size': 100,
            'page': 0,
        }

        while True:
            from metrika.pylib.http import request
            escalations = request('https://juggler-api.search.yandex.net/v2/escalations/get_escalations_log', json=payload, method='POST').json()['escalations']
            _break = False
            for e in escalations:
                if not e['escalation_id']:
                    raise Exception()
                if start_time <= e['start_time'] <= end_time:
                    result[e['escalation_id']] = e
                else:
                    _break = True

            if _break:
                break

            payload['page'] += 1

        return result

    def _get_stat_table(self, stats_dict, name, sort_by_total=False):
        total = defaultdict(int)
        if sort_by_total:
            stats_dict = sorted(stats_dict.items(), key=lambda x: x[1]['night'] + x[1]['day'] + x[1]['manual'], reverse=True)
        else:
            stats_dict = sorted(stats_dict.items())
        data = '#|\n'
        data += '||{}||\n'.format(' | '.join('**{}**'.format(header) for header in [name, 'TOTAL', 'NIGHT', 'DAY', 'MANUAL']))
        for key, stats in stats_dict:
            for k, v in stats.items():
                total[k] += v
            data += '||{}||\n'.format(' | '.join(str(x) for x in [
                key,
                stats['night'] + stats['day'] + stats['manual'],
                stats['night'],
                stats['day'],
                stats['manual'],
            ]))
        data += '||{}||\n'.format(' | '.join('**{}**'.format(x) for x in [
            'TOTAL',
            total['night'] + total['day'] + total['manual'],
            total['night'],
            total['day'],
            total['manual'],
        ]))
        data += '|#'
        return data

    def _get_escalations_table(self, escalations):
        data = '#|\n'
        data += '||{}||\n'.format(' | '.join('**{}**'.format(header) for header in ['Start', 'host:service', 'Stopped', 'Day/Night', 'Login', 'Description']))
        for e in sorted(escalations, key=lambda e: e['started']):
            data += '||{}||\n'.format(' | '.join([
                e['started'].strftime('%Y-%m-%d %H:%M %a'),
                e['key'],
                e['stopped'].strftime('%Y-%m-%d %H:%M %a'),
                e['mark'],
                e['login'],
                e['description']
            ]))
        data += '|#'
        return data

    def escalations(self):
        escalations_tags = self.ESCALATIONS_TAGS.get(self.Parameters.group)
        if not escalations_tags:
            return

        end_date = datetime.now()
        start_date = end_date - timedelta(days=self.Parameters.duration)

        escalations = self._get_escalations(
            filters=[
                {'tags': [tag], 'namespace': 'metrika'} for tag in escalations_tags
            ],
            start_time=mktime(start_date.timetuple()),
            end_time=mktime(end_date.timetuple()),
        )

        events_stat = defaultdict(lambda: {
            'night': 0,
            'day': 0,
            'manual': 0,
        })
        logins_stat = defaultdict(lambda: {
            'night': 0,
            'day': 0,
            'manual': 0,
        })
        days_stat = defaultdict(lambda: {
            'night': 0,
            'day': 0,
            'manual': 0,
        })

        calls = set()
        events = []
        for e in escalations.values():
            if e['end_reason'] != 'USER_STOPPED':
                continue

            login = e['stopped']['login']
            stopped_time = datetime.fromtimestamp(e['stopped']['time'])
            description = e['stopped']['description'].replace('\n', ' ')

            key = '{}:{}'.format(e['host'], e['service'])
            day_key = stopped_time.strftime('%Y-%m-%d %a')
            call_key = stopped_time.strftime('%Y-%m-%d %H:%M:%S')

            mark = ''
            if description == 'Stopped by phone':
                if stopped_time.hour >= 23 or stopped_time.hour <= 8:
                    mark = 'NIGHT'
                    events_stat[key]['night'] += 1
                    if call_key not in calls:
                        days_stat[day_key]['night'] += 1
                        logins_stat[login]['night'] += 1
                        calls.add(call_key)
                else:
                    mark = 'DAY'
                    events_stat[key]['day'] += 1
                    if call_key not in calls:
                        days_stat[day_key]['day'] += 1
                        logins_stat[login]['day'] += 1
                        calls.add(call_key)
            else:
                events_stat[key]['manual'] += 1
                days_stat[day_key]['manual'] += 1
                logins_stat[login]['manual'] += 1

            events.append({
                'key': key,
                'started': datetime.fromtimestamp(e['start_time']),
                'stopped': stopped_time,
                'login': login,
                'mark': mark,
                'description': description,
            })

        issue = self.st_client.issues[self.Context.previous_ticket_key]
        comment = ''
        for description, text in [
            ('Все события', self._get_escalations_table(events)),
            ('Статистика по событиям', self._get_stat_table(events_stat, 'Check', sort_by_total=True)),
            ('Статистика по дням', self._get_stat_table(days_stat, 'Day')),
            ('Статистика по людям', self._get_stat_table(logins_stat, 'Login', sort_by_total=True)),
        ]:
            comment += '<{{{description}\n{text}\n}}>'.format(description=description, text=text)

        issue.comments.create(text=comment)
