# coding=utf-8
import json
import logging

import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry
from six.moves.urllib import parse

from sandbox.projects.metrika.utils.auth import OAuth


class Method(object):

    def __init__(self, token, base_url="https://infra-api.yandex-team.ru/v1/"):
        retries = 3
        backoff_factor = 0.3,
        status_forcelist = (500, 502, 504)
        adapter = HTTPAdapter(max_retries=Retry(total=retries, read=retries, connect=retries, backoff_factor=backoff_factor, status_forcelist=status_forcelist))

        self._session = requests.session()
        self._session.auth = OAuth(token)
        self._session.headers = {"Content-Type": "application/json"}
        self._session.mount("http://", adapter)
        self._session.mount("https://", adapter)

        self._base_url = base_url

    def _url(self, path):
        return parse.urljoin(self._base_url, path)

    def _get(self, path, params=None):
        return self._request("GET", path, params=params)

    def _post(self, path, data):
        return self._request("POST", path, data)

    def _put(self, path, data):
        return self._request("PUT", path, data)

    def _delete(self, path):
        return self._request("DELETE", path)

    def _request(self, method, path, data=None, params=None):
        url = self._url(path)
        logging.info("Request from {}".format(url))

        if data:
            data = json.dumps(data)
            logging.info("Request data {}".format(data))

        if params:
            logging.info("Request params {}".format(params))

        response = self._session.request(method=method, url=url, data=data, params=params)
        if response.status_code != 200:
            logging.exception(response.text)
        response.raise_for_status()
        return response.json()
