# coding=utf-8
from __future__ import unicode_literals

import logging
import re
import time
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.common.errors import UnknownTaskType
from sandbox.common.urls import get_task_link
from sandbox.projects.common import binary_task
from sandbox.projects.metrika.admins.infra.api.infra_api import InfraApi
from sandbox.projects.metrika.admins.infra import metrika_infra_upload
from sandbox.projects.metrika.admins.metrika_admin_package import MetrikaAdminPackage
from sandbox.projects.metrika.admins.metrika_cms_release import MetrikaCmsRelease
from sandbox.projects.metrika.core.metrika_core_release import MetrikaCoreRelease
from sandbox.projects.metrika.frontend.metrika_frontend_release import MetrikaFrontendRelease
from sandbox.projects.metrika.java.metrika_java_release import MetrikaJavaRelease
from sandbox.projects.metrika.utils import settings
from sandbox.projects.metrika.admins.infra.metrika_infra_upload import MetrikaInfraUpload
from sandbox.projects.metrika.admins.infra.metrika_infra_upload.event import Event
from sandbox.projects.metrika.utils import CommonParameters
from sandbox.projects.metrika.utils.base_metrika_task import with_parents, BaseMetrikaTask
from sandbox.projects.metrika.utils.mixins.juggler_reporter import JugglerReporterMixin
from sandbox.projects.metrika.utils.task_types.ya_package import MetrikaYaPackage

TASK_SERVICE = {
    MetrikaCoreRelease: 'Metrika Core releases',
    MetrikaJavaRelease: 'Metrika Java releases',
    MetrikaFrontendRelease: 'Metrika Frontend releases',
    MetrikaAdminPackage: 'Metrika Infra releases',
    MetrikaCmsRelease: 'Metrika Infra releases',
}

REGISTRY_SERVICE = {
    'frontend': 'Metrika Frontend releases',
    'java': 'Metrika Java releases',
    'metrika': 'Metrika Infra releases',
}

DIR_SERVICES = {
    'frontend': 'Metrika Frontend releases',
    'java': 'Metrika Java releases',
    'admin': 'Metrika Infra releases',
    'core': 'Metrika Core releases',
}


@with_parents
class MetrikaInfraDeployUpload(BaseMetrikaTask, JugglerReporterMixin):
    class Parameters(CommonParameters):
        datetime = sdk2.parameters.String('Дата', description='Дата, начиная с которой будут загружаться события')

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def _juggler_warn_predicate(self):
        return self.Context.errors

    def get_event(self):
        return None, None, None, str(self.Context.errors)

    def on_execute(self):
        from metrika.pylib.deploy.client import DeployAPI
        self.Context.errors = []

        if self.Parameters.datetime:
            datetime_from = datetime.strptime(self.Parameters.datetime, metrika_infra_upload.DATETIME_FORMAT)
        else:
            # last hour
            datetime_from = (datetime.now() - timedelta(hours=1)).replace(minute=0, second=0, microsecond=0)
        timestamp_from = int(time.mktime(datetime_from.timetuple()))

        infra_api = InfraApi(sdk2.yav.Secret(settings.yav_uuid).data()['infra-token'])
        namespace_id = MetrikaInfraUpload.get_namespace_id(infra_api, 'Metrika')

        deploy_api = DeployAPI(token=sdk2.yav.Secret(settings.yav_uuid).data()['deploy-token'])
        releases = deploy_api.yp_client.select_objects(
            b'release',
            filter=(
                '[/meta/author_id] = "robot-metrika-test" and '
                '[/status/progress/closed/status] = "true" and '
                '[/status/progress/end_time/seconds] >= {}'.format(timestamp_from)
            ),
            selectors=['']
        )

        for r in releases:
            release = r[0]
            logging.debug('Parsing release: %s', release)
            release_id = release['meta']['id']
            release_rule_id = deploy_api.yp_client.select_objects(
                b'deploy_ticket', filter='[/spec/release_id] = "{}"'.format(release_id), selectors=['/spec/release_rule_id'], limit=1
            )
            if not release_rule_id or not release_rule_id[0]:
                continue
            release_rule_id = release_rule_id[0][0]
            stage_id = deploy_api.yp_client.select_objects(
                b'release_rule', filter='[/meta/id] = "{}"'.format(release_rule_id), selectors=['/meta/stage_id'], limit=1
            )
            if not stage_id or not stage_id[0]:
                continue
            stage_id = stage_id[0][0]

            start = release['status']['progress']['start_time']['seconds']
            end = release['status']['progress']['end_time']['seconds']

            ticket = re.search(r'^[A-Z]+-\d+\b', r[0]['spec'].get('title', ''))
            if ticket:
                ticket = ticket.group()

            release_issues = None
            if 'sandbox' in release['spec']:
                sandbox_spec = release['spec']['sandbox']
                author = sandbox_spec['release_author']
                env = sandbox_spec['release_type']

                try:
                    sb_task = sdk2.Task[sandbox_spec['task_id']]
                except UnknownTaskType as e:
                    logging.warning(str(e))
                    continue

                if sb_task.type in TASK_SERVICE:
                    sb_task = sdk2.Task[sb_task.parent]
                    if not ticket:
                        ticket = getattr(sb_task.Parameters, 'tracker_issue', None)
                    release_issues = sb_task.Context.release_issue_key
                    if not isinstance(release_issues, list):
                        release_issues = [release_issues]

                    service = TASK_SERVICE[sb_task.type]

                    programs = []
                    versions = []
                    for resource in sandbox_spec['resources']:
                        resource_attrs = resource['attributes']
                        programs.append(resource_attrs['resource_name'])
                        versions.append(resource_attrs['resource_version'])
                elif sb_task.type == MetrikaYaPackage:
                    package = sb_task.Parameters.packages.split(';')[0].strip()  # metrika/{{project}}/.../{{program}}/tarball.json
                    service = DIR_SERVICES[package.split('/')[1]]
                    programs, versions = zip(*sb_task.Context.output_resources.items())
                else:
                    logging.warning('Unknown task type {}'.format(sb_task))
                    continue

                programs_comment = '\n'.join(
                    '🗄️ {}={}'.format(program, version)
                    for program, version in zip(programs, versions)
                )
            elif 'docker' in r[0]['spec']:
                sb_task = None
                docker_spec = release['spec']['docker']
                author = docker_spec['release_author']

                env = r[0]['spec']['docker']['release_type']

                if 'images' in docker_spec:
                    images = [i['name'] for i in docker_spec['images']]
                    versions = [i['tag'] for i in docker_spec['images']]
                    registries = [i['registry_host'] for i in docker_spec['images']]
                else:
                    images = [docker_spec['image_name']]
                    versions = [docker_spec['image_tag']]
                    registries = [docker_spec['registry']]

                project = images[0].split('/')[-2:][0]
                programs = [i.split('/')[-1] for i in images]
                programs_comment = '\n'.join(
                    '🐳 {}/{}={}'.format(registry, program, version)
                    for registry, program, version in zip(registries, programs, versions)
                )
                service = REGISTRY_SERVICE[project]
            else:
                self.Context.errors.append('Unknown release type, spec: {}'.format(release))
                continue

            if env not in ['testing', 'stable']:
                continue
            if env == 'stable':
                env = 'production'

            description = (
                '🎫 https://deploy.yandex-team.ru/stage/{}/releases/{}\n\n'.format(stage_id, release_id) +
                '👨 https://staff.yandex-team.ru/{}\n\n'.format(author) +
                '{}\n\n'.format(programs_comment) +
                ('Создан в Sandbox: {}\n'.format(get_task_link(sb_task.id)) if sb_task else '') +
                (
                    ('Релизные тикеты:\n ' + '\n'.join('http://st/{}'.format(ri) for ri in release_issues) + '\n')
                    if release_issues
                    else ''
                ) +
                ('Тикет: http://st/{}'.format(ticket) if ticket else '')
            )
            logging.debug('%s %s', service, env)

            event = Event(
                title='Выкладка ' + ', '.join(programs),
                description=description,
                service_id=MetrikaInfraUpload.get_service_id(infra_api, namespace_id, service),
                environment_id=MetrikaInfraUpload.get_environment_id(infra_api, namespace_id, service, env),
                start_time=start,
                finish_time=end,
                type='maintenance',
                severity='minor',
                tickets=ticket,
            )
            event.id = infra_api.events.create(event.to_dict())['id']
            self.set_info('<a href="https://infra.yandex-team.ru/event/{}">{}</a>'.format(event.id, event.title), do_escape=False)
