# coding=utf-8
import re
import time
from datetime import datetime, timedelta
from itertools import groupby
from operator import itemgetter

import json
from sandbox.projects.common import binary_task
from sandbox.projects.common.decorators import memoize
from sandbox.projects.metrika.admins.infra.api.infra_api import InfraApi
from sandbox.projects.metrika.admins.infra.metrika_infra_upload.event import Event
from sandbox.projects.metrika.admins.infra.metrika_infra_upload.ticket import Ticket
from sandbox.projects.metrika.utils import CommonParameters
from sandbox.projects.metrika.utils import settings
from sandbox.projects.metrika.utils.base_metrika_task import with_parents, BaseMetrikaTask
from sandbox.projects.metrika.utils.mixins.juggler_reporter import JugglerReporterMixin
from sandbox.sdk2 import Task
from sandbox.sdk2 import Vault
from sandbox.sdk2.parameters import String

DATETIME_FORMAT = "%Y-%m-%d %H:%M:%S"
DATE_FORMAT = "%Y-%m-%d"
CONDUCTOR_SERVICE = {
    "metrika": "Metrika Core releases",
    "metrika-trusty": "Metrika Core releases",
    "metrika-common": "Metrika Java releases",
    "verstka": "Metrika Frontend releases",
    "yandex-precise": "Metrika Infra releases",
    "yandex-trusty": "Metrika Infra releases",
    "yandex-xenial": "Metrika Infra releases",
    "mtrs-common": "Metrika Infra releases",
    "common": "Metrika Infra releases",
    "search": "Metrika Infra releases",
}
EVENT_ENVIRONMENT = {
    "testing": "testing",
    "stable": "production",
    "hotfix": "production"
}
EVENT_SEVERITY = {
    "testing": "minor",
    "stable": "minor",
    "hotfix": "major"
}


@with_parents
class MetrikaInfraUpload(BaseMetrikaTask, JugglerReporterMixin, Task):
    """
    Загрузка информации о релизах в Инфру
    """

    class Parameters(CommonParameters):
        description = "Загрузка информации о релизах в Инфру"

        environment = String("Окружение", required=True, default="testing", choices=[("testing", "testing"), ("production", "production")],
                             description="Окружение, для которого будут загружаться события.")

        date = String("Дата", required=False, default=datetime.now().strftime(DATE_FORMAT), description="Дата, за которую будут загружаться события.")

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def on_execute(self):
        date = self.Parameters.date or datetime.now().strftime(DATE_FORMAT)

        tickets = self.get_tickets(self.Parameters.environment, date)

        self.set_tickets_info(tickets)

        events = self.get_events(date)

        new_tickets = self.filter_new_tickets(tickets, events)
        new_events = self.get_events_from_tickets(new_tickets)

        self.set_new_events_info(new_events)

        self.upload_events(new_events)

        self.update_new_events_info(new_events)

        changed_tickets = self.filter_changed_tickets(tickets, events)
        changed_events = self.update_events_from_tickets(events, changed_tickets)

        self.set_changed_events_info(changed_events)

        self.update_events(changed_events)

    def set_tickets_info(self, tickets):
        self.Context.tickets = [ticket.__dict__ for ticket in tickets]

        if tickets:
            tickets_info = "Кондукторные тикеты:<br/>" + "\n".join("<a href=\"https://c.yandex-team.ru/tickets/{0}\">{0}</a> {1}".format(
                ticket.id, ", ".join("{}={}".format(package.get("name"), package.get("version")) for package in ticket.packages)
            ) for ticket in tickets)
            self.set_info(tickets_info, do_escape=False)
        else:
            self.set_info("Кондукторные тикеты отсутствуют", do_escape=False)

    def set_new_events_info(self, events):
        self.Context.new_events = [event.__dict__ for event in events]

    def update_new_events_info(self, events):
        self.set_new_events_info(events)

        if events:
            events_info = "Добавленные события:<br/>" + "\n".join("<a href=\"https://infra.yandex-team.ru/event/{0}\">{0}</a> {1}".format(event.id, event.title) for event in events)
            self.set_info(events_info, do_escape=False)
        else:
            self.set_info("Новых событий не добавлено", do_escape=False)

    def set_changed_events_info(self, events):
        self.Context.changed_events = [event.__dict__ for event in events]

        if events:
            events_info = "Обновленные события:<br/>" + "\n".join("<a href=\"https://infra.yandex-team.ru/event/{0}\">{0}</a> {1}".format(event.id, event.title) for event in events)
            self.set_info(events_info, do_escape=False)
        else:
            self.set_info("Существующие события не обновлялись", do_escape=False)

    @staticmethod
    def get_tickets(environment, date):
        from metrika.pylib.conductor.conductor import ConductorAPI
        conductor_api = ConductorAPI(timeout=30)
        tasks = [task for tasks in [
            conductor_api.tasks_filter("metrika", key, date, date) + conductor_api.tasks_filter("yabs", key, date, date) for key, value in EVENT_ENVIRONMENT.iteritems() if value == environment
        ] for task in tasks]
        return [MetrikaInfraUpload.get_ticket(conductor_api, key, list(group)) for key, group in groupby(sorted(tasks, key=itemgetter("ticket")), itemgetter("ticket"))]

    @staticmethod
    def get_ticket(api, ticket_id, tasks):
        last_task = max(tasks, key=itemgetter("done_at"))
        return Ticket(ticket_id, api.ticket(ticket_id).status,
                      MetrikaInfraUpload.get_ticket_service(api, ticket_id),
                      MetrikaInfraUpload.get_ticket_started_at(api, ticket_id, last_task.get("done_at")),
                      last_task.get("done_at"), last_task.get("name"))

    @staticmethod
    def get_ticket_service(api, ticket_id):
        ticket = api.ticket(ticket_id).status
        package = MetrikaInfraUpload.get_package_id(api, ticket.get("packages")[0].get("name"))
        repository = [repo.get("attributes").get("name") for repo in MetrikaInfraUpload.conductor_request(api, "/api/v2/packages/{}/repos".format(package)).get("data")][0]
        return CONDUCTOR_SERVICE.get(repository)

    @staticmethod
    def get_package_id(api, package_name):
        return package_name if "." not in package_name else MetrikaInfraUpload.conductor_request(api, "/api/v1/packages/{}".format(package_name)).get("id")

    @staticmethod
    def get_ticket_started_at(api, ticket_id, limit):
        tasks = MetrikaInfraUpload.conductor_request(api, "/api/v2/tickets/{}/tasks".format(ticket_id)).get("data")
        tasks_logs = [MetrikaInfraUpload.get_task_logs(api, task) for task in tasks]
        tasks_dates = [logs[0].get("date") for logs in tasks_logs if logs]
        if tasks_dates:
            started_at = min(tasks_dates).replace("T", " ")
            if MetrikaInfraUpload.get_event_timestamp(started_at) <= MetrikaInfraUpload.get_event_timestamp(limit):
                return started_at
        return api.ticket(ticket_id).status.get("created_at")

    @staticmethod
    def get_task_logs(api, task):
        return api.task(task.get("attributes").get("name"))._get_log()

    @staticmethod
    def conductor_request(api, path):
        return json.loads(api._request(api._url + path, headers={"Authorization": "OAuth {}".format(Vault.data(settings.owner, settings.conductor_token))}))

    @staticmethod
    def get_events(date):
        infra_api = InfraApi(Vault.data(settings.owner, settings.infra_token))

        start_timestamp = MetrikaInfraUpload.get_timestamp(datetime.strptime(date, DATE_FORMAT))
        end_timestamp = MetrikaInfraUpload.get_timestamp(datetime.strptime(date, DATE_FORMAT) + timedelta(1))

        namespace_id = MetrikaInfraUpload.get_namespace_id(infra_api, "Metrika")
        all_service_ids = [service.get("id") for service in infra_api.namespaces.get_services(namespace_id)]
        return [Event.from_dict(item) for sublist in
                [infra_api.events.get_all(from_timestamp=start_timestamp, to_timestamp=end_timestamp, service_id=service_id) for service_id in all_service_ids]
                for item in sublist]

    @staticmethod
    def filter_new_tickets(tickets, events):
        uploaded_ticket_ids = [event.meta.get("ticket_id") for event in events]
        return [ticket for ticket in tickets if ticket.id not in uploaded_ticket_ids]

    @staticmethod
    def get_events_from_tickets(tickets):
        infra_api = InfraApi(Vault.data(settings.owner, settings.infra_token))

        namespace_id = MetrikaInfraUpload.get_namespace_id(infra_api, "Metrika")

        return [Event(
            title=MetrikaInfraUpload.get_event_title(ticket),
            description=MetrikaInfraUpload.get_event_description(ticket),
            service_id=MetrikaInfraUpload.get_service_id(infra_api, namespace_id, ticket.service),
            environment_id=MetrikaInfraUpload.get_environment_id(infra_api, namespace_id, ticket.service, EVENT_ENVIRONMENT.get(ticket.branch)),
            start_time=MetrikaInfraUpload.get_event_timestamp(ticket.started_at),
            finish_time=MetrikaInfraUpload.get_event_timestamp(ticket.done_at),
            type="maintenance",
            severity=EVENT_SEVERITY.get(ticket.branch),
            tickets=MetrikaInfraUpload.get_event_tickets(ticket),
            meta={"ticket_id": ticket.id, "ticket_last_task": ticket.last_task}
        ) for ticket in tickets]

    @staticmethod
    def get_event_title(ticket):
        return "Выкладка {}".format(", ".join(MetrikaInfraUpload.get_daemon_name(package.get("name")) for package in ticket.packages))

    @staticmethod
    def get_daemon_name(package):
        return package.replace("-metrika-yandex", "").replace("yandex-metrika-", "").replace("-yandex", "").replace("yandex-", "")

    @staticmethod
    def get_event_description(ticket):
        return "🎫 https://c.yandex-team.ru/tickets/{}\n\n".format(ticket.id) + \
               "👨 https://staff.yandex-team.ru/{}\n\n".format(ticket.author) + \
               "\n".join("📦 {}={}".format(package.get("name"), package.get("version")) for package in ticket.packages) + \
               "\n\n" + ticket.comment

    @staticmethod
    @memoize
    def get_namespace_id(api, name):
        return next(namespace.get("id") for namespace in api.namespaces.get_all() if namespace.get("name") == name)

    @staticmethod
    @memoize
    def get_service_id(api, namespace_id, service_name):
        return next(service.get("id") for service in api.namespaces.get_services(namespace_id) if service.get("name") == service_name)

    @staticmethod
    @memoize
    def get_environment_id(api, namespace_id, service_name, environment_name):
        service_id = MetrikaInfraUpload.get_service_id(api, namespace_id, service_name)
        return next(environment.get("id") for environment in api.services.get_mine(service_id).get("environments") if environment.get("name") == environment_name)

    @staticmethod
    def get_event_timestamp(date_time):
        return MetrikaInfraUpload.get_timestamp(datetime.strptime(date_time[:19], DATETIME_FORMAT))

    @staticmethod
    def get_timestamp(date_time):
        return int(time.mktime(date_time.timetuple()))

    @staticmethod
    def get_event_tickets(ticket):
        return ",".join(re.findall(r".*?([A-Z]+-\d+).*?", ticket.comment))

    @staticmethod
    def upload_events(events):
        infra_api = InfraApi(Vault.data(settings.owner, settings.infra_token))
        for event in events:
            event.id = infra_api.events.create(event.to_dict()).get("id")

    @staticmethod
    def filter_changed_tickets(tickets, events):
        uploaded_ticket_ids = [event.meta.get("ticket_id") for event in events]
        uploaded_ticket_last_tasks = [event.meta.get("ticket_last_task") for event in events]
        return [ticket for ticket in tickets if ticket.id in uploaded_ticket_ids and ticket.last_task not in uploaded_ticket_last_tasks]

    @staticmethod
    def update_events_from_tickets(events, tickets):
        return [MetrikaInfraUpload.get_updated_event(next(event for event in events if event.meta.get("ticket_id") == ticket.id), ticket) for ticket in tickets]

    @staticmethod
    def get_updated_event(event, ticket):
        event.finish_time = MetrikaInfraUpload.get_event_timestamp(ticket.done_at)
        event.meta["ticket_last_task"] = ticket.last_task
        return event

    @staticmethod
    def update_events(events):
        infra_api = InfraApi(Vault.data(settings.owner, settings.infra_token))
        for event in events:
            infra_api.events.update(event.id, event.to_dict())
