# coding=utf-8
from __future__ import unicode_literals

from sandbox import sdk2
from sandbox.common import patterns
from sandbox.common.types.task import ReleaseStatus
from sandbox.common.urls import get_task_link
from sandbox.projects.common import conductor, binary_task
from sandbox.projects.common.build.YaPackage2 import YaPackage2Parameters
from sandbox.projects.common.build.ya_package_config.consts import PackageType
from sandbox.projects.common.ya_deploy import release_integration
from sandbox.projects.metrika.admins import programs_list
from sandbox.projects.metrika.utils import settings
from sandbox.projects.metrika.utils.conductor import Conductor
from sandbox.projects.metrika.utils.pipeline.pipeline import PipelineBaseTask
from sandbox.projects.metrika.utils.contextable import Contextable
from sandbox.projects.metrika.utils.parameters import TrackerIssue, ArcadiaURL, choices
from sandbox.projects.metrika.utils.resource_types import daemons
from sandbox.projects.metrika.utils import print_ticket, CommonParameters
from sandbox.projects.metrika.utils import programs
from sandbox.projects.metrika.utils.base_metrika_task import with_parents
from sandbox.projects.metrika.utils.mixins.releasable import ReleasableMixin
from sandbox.projects.metrika.utils.task_types.ya_package import MetrikaYaPackage

PACKAGE_TO_STAGE = {
    'bishop-frontend': 'bishop',
    'clickhouse-backups-frontend': 'clickhouse-backups',
    'duty-frontend': 'duty',
    'zooface-frontend': 'zooface',
    'clickhouse-rbac-frontend': 'clickhouse-rbac',
}


class PackageInfo(Contextable):
    def __init__(self, type=None, config_path=None, version=None, c_ticket=None):
        self.type = type
        self.config_path = config_path
        self.version = version
        self.c_ticket = c_ticket


@with_parents
class MetrikaAdminPackage(PipelineBaseTask, ReleasableMixin):
    PARAMETERS_MAPPING = {
        'build_debian': (programs_list.PACKAGE_JSON, PackageType.DEBIAN.value),
        'build_docker': (programs_list.DOCKER_JSON, PackageType.DOCKER.value),
        'build_tarball': (programs_list.TARBALL_JSON, PackageType.TARBALL.value),
    }

    class Parameters(CommonParameters):
        name = sdk2.parameters.String('Название пакета/образа', required=True, choices=choices(programs.ADMIN_PROGRAMS, autocomplete=True))
        arcadia_url = ArcadiaURL()

        ticket = TrackerIssue()

        create_ticket = sdk2.parameters.Bool('Создать релизный тикет', default=False)

        with sdk2.parameters.Group('Build') as build_group:
            package_resource_type = sdk2.parameters.String('Тип ресурса пакета в YaMake', required=True, default='auto')

            build_debian = sdk2.parameters.Bool('Debian', default=True)
            with build_debian.value[True]:
                package_repo = sdk2.parameters.String('Репозиторий пакета', required=True, default='mtrs-common')

            build_docker = sdk2.parameters.Bool('Docker', default=True)
            with build_docker.value[True]:
                image_repo = sdk2.parameters.String('Репозиторий образа', required=True, default='metrika')

            build_tarball = sdk2.parameters.Bool('Tarball', default=True)

            with build_debian.value[False], build_docker.value[False], build_tarball.value[False]:
                with sdk2.parameters.Group('Выбери хотя бы один тип сборки :(') as build_warning:
                    pass

        with sdk2.parameters.Group('Deploy') as deploy_group:
            auto_release = sdk2.parameters.Bool('Релизнуть после сборки', default=False)
            with auto_release.value[True]:
                release_branches = sdk2.parameters.CheckGroup(
                    'Ветки релиза', required=True, default=ReleaseStatus.TESTING,
                    choices=choices([ReleaseStatus.TESTING, ReleaseStatus.STABLE])
                )

        with build_debian.value[True]:
            with sdk2.parameters.Group('Conductor') as conductor_group:
                auto_conductor = sdk2.parameters.Bool('Выложить после сборки', default=False)
                with auto_conductor.value[True]:
                    conductor_branches = sdk2.parameters.CheckGroup(
                        'Ветки в кондукторе', required=True, default='testing',
                        choices=choices(conductor.BRANCHES)
                    )

        _binary = binary_task.binary_release_parameters_list(stable=True)

    class Context(PipelineBaseTask.Context):
        packages = []
        build_subtasks = []

    @patterns.singleton_property
    def packages(self):
        self.Context.packages = [PackageInfo.fill(p) for p in self.Context.packages]
        return self.Context.packages

    def on_create(self):
        if self.author == 'robert':
            self.Parameters.auto_release = True
            self.Parameters.release_branches = [ReleaseStatus.TESTING, ReleaseStatus.STABLE]
        elif self.author == 'velom':
            self.Parameters.auto_release = True

    def on_save(self):
        if self.Parameters.name:
            self.Parameters.tags = [self.Parameters.name]
        if not (self.Parameters.build_debian or self.Parameters.build_docker or self.Parameters.build_tarball):
            raise ValueError('Choose at least one build type')

    def on_enqueue(self):
        self.Context.arcadia_url = self.Parameters.arcadia_url.format(author=self.author, ticket=self.Parameters.ticket)

    def _get_task_params(self, package_type, config_path):
        if self.Parameters.package_resource_type == 'auto':
            package_resource_type = daemons.get_daemon_resource(self.Parameters.name).name
        else:
            package_resource_type = self.Parameters.package_resource_type

        params = {
            'description': '{} build of {} from {}'.format(package_type, self.Parameters.name, self.Context.arcadia_url),
            YaPackage2Parameters.checkout_arcadia_from_url: self.Context.arcadia_url,
            sdk2.Parameters.tags.name: [self.Parameters.name, package_type],

            YaPackage2Parameters.packages: config_path,
            YaPackage2Parameters.package_type: package_type,
            YaPackage2Parameters.resource_type: package_resource_type,
            YaPackage2Parameters.package_resource_attrs: {
                'program': self.Parameters.name,
                'type': package_type
            },

            release_integration.ReleaseToYaDeployParameter.name: True,
            release_integration.YpTokenVaultParameter.name: settings.yp_token
        }

        if package_type == PackageType.DOCKER.value:
            params.update({
                YaPackage2Parameters.docker_push_image: True,
                YaPackage2Parameters.docker_image_repository: self.Parameters.image_repo,
                YaPackage2Parameters.docker_user: settings.login,
                YaPackage2Parameters.docker_token_vault_name: settings.docker_registry_token
            })
        elif package_type == PackageType.DEBIAN.value:
            params.update({
                YaPackage2Parameters.changelog: self.Parameters.description,
                YaPackage2Parameters.publish_package: True,
                YaPackage2Parameters.publish_to: self.Parameters.package_repo,
                YaPackage2Parameters.key_user: 'robot-metrika-admin'
            })

        return params

    def create_stages(self):
        return [
            (self.create_ticket, 'Трекер'),
            (self.build, 'Сборка'),
            (self.deploy, 'Выкатка'),
        ]

    def create_ticket(self):
        if self.Parameters.create_ticket:
            self.Context.ticket = self.st_client.issues.create(
                queue='MTRSADMIN',
                assignee=self.author,
                summary='Релиз {}'.format(self.Parameters.name),
                components=[77185],
                links=[{
                    'relationship': 'relates',
                    'origin': 'ru.yandex.sandbox',
                    'key': self.id
                }]
            ).key
            if self.Parameters.ticket:
                self.st_client.issues[self.Context.ticket].links.create('relates', self.Parameters.ticket)

            print_ticket(self, self.Context.ticket, comment='Релизный тикет')

        if self.Parameters.ticket:
            print_ticket(self, self.Parameters.ticket)

    def build(self):
        if not self.packages:
            all_programs = programs_list.MetrikaProgramsList.get_all_programs(self.Context.arcadia_url, 'metrika/admin', replace_suffix=False)
            if self.Parameters.name not in all_programs:
                raise Exception('{} files are not found in metrika/admin on {} for program {}'.format(
                    ', '.join(programs_list.CONFIG_FILES),
                    self.Context.arcadia_url,
                    self.Parameters.name
                ))

            for param, (config_name, build_type) in self.PARAMETERS_MAPPING.items():
                if getattr(self.Parameters, param):
                    config_path = all_programs[self.Parameters.name].get(config_name)
                    if config_path:
                        self.packages.append(PackageInfo(type=build_type, config_path=config_path))
                    else:
                        self.set_info('<font color="orange">{} wasn\'t found, skip building {}</font>'.format(config_name, build_type), do_escape=False)

        if self.packages:
            self.run_subtasks(
                [
                    (MetrikaYaPackage, self._get_task_params(info.type, info.config_path))
                    for info in self.packages
                ],
                subtasks_variable=self.Context.build_subtasks
            )
        else:
            raise ValueError('Nothing to build for {}, check config files names'.format(self.Parameters.name))

        for info, subtask_id in zip(self.packages, self.Context.build_subtasks):
            info.version = sdk2.Task[subtask_id].Context.output_resource_version
            self.set_info('<a href="{}">{}</a>: {}={}'.format(get_task_link(subtask_id), info.type.title(), self.Parameters.name, info.version), do_escape=False)

    def conductor(self, branch):
        for info in self.packages:
            if info.type == PackageType.DEBIAN.value:
                with self.memoize_stage["conductor-" + branch](commit_on_entrance=False):
                    info.c_ticket = Conductor(self).create_conductor_ticket(
                        packages={self.Parameters.name: info.version},
                        branch=branch,
                        release_issue_key=self.Context.ticket,
                        issue_key=self.Parameters.ticket,
                        no_autoinstall=True
                    )
                    self.set_info(
                        'Кондукторный тикет в {}: <a href="https://c.yandex-team.ru/tickets/{}">{}={}</a>'.format(
                            branch, info.c_ticket, self.Parameters.name, info.version
                        ),
                        do_escape=False
                    )

    def get_deploy_released_info_message(self, branch):
        base_message = 'Released in deploy to {}'.format(branch)
        if self.author == 'robert':
            name = self.Parameters.name
            if branch == 'stable':
                env = 'production'
            else:
                env = 'testing'
            stage = '{}-{}'.format(PACKAGE_TO_STAGE.get(name, name), env)

            href = '<a href="https://deploy.yandex-team.ru/stage/{}/releases" target="_blank">{}</a>'.format(stage, stage)
            base_message = '{}: {}'.format(base_message, href)
        return base_message

    def deploy(self):
        if self.Parameters.auto_conductor:
            for branch in self.Parameters.conductor_branches:
                self.conductor(branch=branch)

        if self.Parameters.auto_release and any(info.type != PackageType.DEBIAN.value for info in self.packages):
            for branch in self.Parameters.release_branches:
                with self.memoize_stage["deploy-" + branch](commit_on_entrance=False):
                    ReleasableMixin.on_release(self, {'release_status': branch})
                    info = self.get_deploy_released_info_message(branch)
                    self.set_info(
                        info,
                        do_escape=False,
                    )

    def on_release(self, parameters):
        self.conductor(branch=parameters['release_status'])
