# coding=utf-8
import logging
import os.path

import sandbox.projects.common.build.parameters as build_parameters
import sandbox.projects.common.utils as common_utils
from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.metrika.utils.task_types.ya_make import MetrikaYaMake
from sandbox.sdk2.helpers import gdb as gdb_helpers


class MetrikaBuildTasklet(MetrikaYaMake):
    """
    Сборка тасклета, на выходе - идентификатор ресурса после загрузки
    """

    class Parameters(MetrikaYaMake.Parameters):
        create_html_results_resource = build_parameters.CreateHtmlResultsParameter(default_value=True)

        ya_make_project_params = None

        tasklet = sdk2.parameters.String("Tasklet target", required=True,
                                         description="Path to tasklet target")
        login = sdk2.parameters.String("Username", required=True, default_value="robot-metrika-test")
        ssh_key = sdk2.parameters.Vault("SSH key in Vault ('name' or 'owner:name')", required=True,
                                        default_value="METRIKA:robot-metrika-test-ssh")

        _binary = binary_task.binary_release_parameters_list(stable=True)

        with sdk2.parameters.Output:
            resource = sdk2.parameters.Resource("Resource with uploaded tasklet")

    def get_targets(self):
        return [self.Parameters.tasklet]

    def get_arts(self):
        return [{"path": os.path.join(self.Parameters.tasklet, self.binary_name)}]

    def get_arts_source(self):
        return []

    def get_resources(self):
        return {
            "project": {
                "description": "Tasklet binary {}".format(self.Parameters.tasklet),
                "resource_path": self.binary_name
            }
        }

    @property
    def binary_name(self):
        return os.path.basename(self.Parameters.tasklet)

    def get_upload_command(self, pack_dir):
        return [
            os.path.join(pack_dir, self.binary_name), "sb-upload",
            "--sb-owner", self.owner, "--sb-schema"
        ]

    def post_build(self, source_dir, output_dir, pack_dir):
        logging.info("Post build - tasklet upload")
        try:
            with sdk2.ssh.Key(self, self.Parameters.ssh_key.owner, self.Parameters.ssh_key.name):
                with sdk2.helpers.ProcessLog(self,
                                             logger="console",
                                             stdout_level=logging.INFO,
                                             stderr_level=logging.ERROR) as pl:
                    try:
                        env = os.environ.copy()
                        env["USER"] = self.Parameters.login
                        output = sdk2.helpers.subprocess.check_output(self.get_upload_command(pack_dir), env=env, stderr=pl.stderr)
                        pl.logger.info(output)
                        self.Parameters.resource = sdk2.Resource[output]
                    except sdk2.helpers.subprocess.CalledProcessError:
                        self.set_info(gdb_helpers.get_html_view_for_logs_file("stdout", pl.stdout.path.relative_to(self.log_path()), self.log_resource), do_escape=False)
                        self.set_info(gdb_helpers.get_html_view_for_logs_file("stderr", pl.stdout.path.relative_to(self.log_path()), self.log_resource), do_escape=False)
                        raise

                    common_utils.set_resource_attributes(self.Parameters.resource, {"released": "stable", "ttl": "inf", "backup": True})
        except Exception as e:
            logging.exception("Exception in postbuild. Defer.")
            self.set_info('Delay exception %s' % str(e))
            self.Context.exceptions = True
