# coding=utf-8
from sandbox import sdk2

from sandbox.common.types.task import ReleaseStatus
from sandbox.common.types.task import Status
from sandbox.projects.common import binary_task
from sandbox.projects.common.build.YaPackage2 import YaPackage2Parameters
from sandbox.projects.common.build.parameters import ArcadiaPatch
from sandbox.projects.metrika.utils import resource_types
from sandbox.projects.metrika.utils import settings
from sandbox.projects.metrika.utils.conductor import Conductor
from sandbox.projects.metrika.utils.pipeline.pipeline import PipelineBaseTask
from sandbox.projects.metrika.utils.parameters import TrackerIssue, ArcadiaURL
from sandbox.projects.metrika.utils.mixins.subtasks import SubTasksFailure
from sandbox.projects.metrika.utils import CommonParameters
from sandbox.projects.metrika.utils.base_metrika_task import with_parents
from sandbox.projects.metrika.utils.mixins.releasable import ReleasableMixin, set_skynet_id_attr
from sandbox.projects.metrika.utils.task_types.ya_package import MetrikaYaPackage


class MetrikaCmsFrontend(resource_types.BaseMetrikaBinaryResource):
    daemon_name = 'cms-frontend'


class MetrikaCmsJudge(resource_types.BaseMetrikaBinaryResource):
    daemon_name = 'cms-judge'


class MetrikaCmsMarshal(resource_types.BaseMetrikaBinaryResource):
    daemon_name = 'cms-marshal'


COMPONENT_GROUP_DEPLOY = "deploy"
COMPONENT_GROUP_CONDUCTOR = "conductor"


@with_parents
class MetrikaCmsRelease(PipelineBaseTask, ReleasableMixin):
    class Parameters(CommonParameters):
        arcadia_url = ArcadiaURL()
        arcadia_patch = ArcadiaPatch()

        ticket = TrackerIssue()

        with sdk2.parameters.RadioGroup('Компоненты') as components_group:
            components_group.values[COMPONENT_GROUP_DEPLOY] = components_group.Value("Frontend, Judge, Marshal", default=True)
            components_group.values[COMPONENT_GROUP_CONDUCTOR] = components_group.Value("CMS-agent")

        with sdk2.parameters.Group('Развёртывание') as deploy_group:
            auto_release_testing = sdk2.parameters.Bool('Testing', default=True)
            auto_release_stable = sdk2.parameters.Bool('Stable', default=True)

        _binary = binary_task.binary_release_parameters_list(stable=True)

    class Context(PipelineBaseTask.Context):
        subtasks_ids = []
        deploy_wait_ids = []
        arcadia_url = None

    def on_enqueue(self):
        self.Context.arcadia_url = self.Parameters.arcadia_url.format(author=self.author, ticket=self.Parameters.ticket)

    def create_stages(self):
        stages = [
            (self.build, 'Сборка'),
        ]

        if self.Parameters.auto_release_testing:
            stages.append((self.deploy_to_testing, 'Развёртывание в testing'))

        if self.Parameters.auto_release_stable:
            stages.append((self.deploy_to_stable, 'Развёртывание в stable'))

        return stages

    @property
    def summary(self):
        components = []
        if self.Parameters.components_group == COMPONENT_GROUP_DEPLOY:
            components.append("frontend")
            components.append("judge")
            components.append("marshal")
        if self.Parameters.components_group == COMPONENT_GROUP_CONDUCTOR:
            components.append("agent")

        return "Релиз CMS. Компоненты: {}".format(", ".join(components))

    @property
    def tags(self):
        tags = ["cms"]
        if self.Parameters.components_group == COMPONENT_GROUP_DEPLOY:
            tags.append("frontend")
            tags.append("judge")
            tags.append("marshal")
        elif self.Parameters.components_group == COMPONENT_GROUP_CONDUCTOR:
            tags.append("agent")
            tags.append("cms-agent")
        if self.Parameters.auto_release_testing:
            tags.append("testing")
        if self.Parameters.auto_release_stable:
            tags.append("stable")
            tags.append("production")
        return tags

    @property
    def tarballs(self):
        return [
            {
                "name": "cms-marshal",
                "path": "metrika/admin/packages/cms/marshal/tarball.json",
                "resource_type": str(MetrikaCmsMarshal)
            },
            {
                "name": "cms-frontend",
                "path": "metrika/admin/packages/cms/frontend/tarball.json",
                "resource_type": str(MetrikaCmsFrontend)
            },
            {
                "name": "cms-judge",
                "path": "metrika/admin/packages/cms/judge/tarball.json",
                "resource_type": str(MetrikaCmsJudge)
            }
        ]

    def build(self):
        self.Parameters.tags = self.tags

        subtasks_params = []

        if self.Parameters.components_group == COMPONENT_GROUP_DEPLOY:
            subtasks_params.append(
                (
                    MetrikaYaPackage,
                    {
                        'description': self.Parameters.description,
                        YaPackage2Parameters.checkout_arcadia_from_url: self.Context.arcadia_url,
                        YaPackage2Parameters.arcadia_patch: self.Parameters.arcadia_patch,
                        sdk2.Parameters.tags.name: ["cms"] + [t["name"] for t in self.tarballs],
                        YaPackage2Parameters.packages: ";".join([t["path"] for t in self.tarballs]),
                        YaPackage2Parameters.package_type: "tarball",
                        YaPackage2Parameters.resource_type: ";".join([t["resource_type"] for t in self.tarballs]),
                        YaPackage2Parameters.release_to_ya_deploy: True,
                        YaPackage2Parameters.yp_token_vault: settings.yp_token
                    }
                )
            )

        if self.Parameters.components_group == COMPONENT_GROUP_CONDUCTOR:
            subtasks_params.append(
                (
                    MetrikaYaPackage,
                    {
                        'description': self.Parameters.description,
                        YaPackage2Parameters.changelog: self.Parameters.description,
                        YaPackage2Parameters.checkout_arcadia_from_url: self.Context.arcadia_url,
                        YaPackage2Parameters.arcadia_patch: self.Parameters.arcadia_patch,
                        sdk2.Parameters.tags.name: ["cms", "agent", "cms-agent"],
                        YaPackage2Parameters.packages: "metrika/admin/packages/cms/agent/package.json",
                        YaPackage2Parameters.package_type: "debian",
                        YaPackage2Parameters.resource_type: str(resource_types.BaseMetrikaBinaryResource),
                        YaPackage2Parameters.debian_distribution: "unstable",
                        YaPackage2Parameters.publish_package: True,
                        YaPackage2Parameters.key_user: "robot-metrika-admin",
                        YaPackage2Parameters.publish_to: "metrika-trusty"
                    }
                )
            )

        self.run_subtasks(subtasks=subtasks_params, subtasks_variable=self.Context.subtasks_ids)

    def deploy_to_testing(self):
        self.deploy(ReleaseStatus.TESTING)

    def deploy_to_stable(self):
        self.deploy(ReleaseStatus.STABLE)

    def deploy(self, to):
        if self.Parameters.components_group == COMPONENT_GROUP_CONDUCTOR:
            with self.memoize_stage["conductor_release_{}".format(to)](commit_on_entrance=True):
                subtask = self.find(task_type=MetrikaYaPackage, status=Status.Group.SUCCEED).first()
                name = "cms-agent"
                version = subtask.Context.output_resource_version
                issue_key = self.Parameters.ticket if self.Parameters.ticket else None
                no_autoinstall = True if to == ReleaseStatus.STABLE else False
                if subtask:
                    conductor_ticket = Conductor(self).create_conductor_ticket(
                        packages={name: version},
                        branch=str(to),
                        release_issue_key=None,
                        issue_key=issue_key,
                        no_autoinstall=no_autoinstall
                    )
                    self.set_info('Кондукторный тикет: <a href="https://c.yandex-team.ru/tickets/{}">{}={}</a>'.format(conductor_ticket, name, version), do_escape=False)

        if self.Parameters.components_group == COMPONENT_GROUP_DEPLOY:
            with self.memoize_stage["initiate_release_{}".format(to)](commit_on_entrance=False):
                subtask = self.find(task_type=MetrikaYaPackage, status=Status.Group.SUCCEED).first()
                if subtask:
                    version = subtask.Context.output_resource_version
                    comment = "{version} by {author} for {ticket}".format(
                        version=version,
                        author=subtask.author,
                        ticket=self.Parameters.ticket if self.Parameters.ticket else "passion"
                    )
                    release_params = {
                        "task_id": subtask.id,
                        "type": to,
                        "subject": "Release of {}".format(", ".join([t["name"] for t in self.tarballs])),
                        "message": comment,
                    }
                    self.server.release(release_params)
                    set_skynet_id_attr(task_id=subtask.id)
                self.Context.deploy_wait_ids = [subtask.id]
                raise sdk2.WaitTask(self.Context.deploy_wait_ids, [Status.RELEASED, Status.NOT_RELEASED])

            failed_tasks = [subtask_id for subtask_id in self.Context.deploy_wait_ids if sdk2.Task[subtask_id].status == Status.NOT_RELEASED]
            if failed_tasks:
                raise SubTasksFailure(failed_tasks)
