# coding=utf-8
import datetime
import logging
import re

from sandbox.projects.common import binary_task
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.utils import base_metrika_task, settings
from sandbox.projects.metrika.utils.mixins import juggler_reporter
from sandbox.sdk2 import parameters


class MetrikaForsakenStandsRemove(base_metrika_task.BaseMetrikaTask, juggler_reporter.JugglerReporterMixin):
    """
    Удаление устаревших тестовых стендов Метрики
    """

    class Parameters(utils.CommonParameters):
        description = "Удаление устаревших тестовых стендов Метрики"

        max_age = parameters.Integer("Максимальнй возраст стенда в днях", required=True, default=14)

        with parameters.Group("Секреты") as secrets_group:
            tokens_secret = parameters.YavSecret("Секрет с токенами", required=True, default=settings.yav_uuid)

            deploy_token_key = parameters.String("Ключ токена Деплоя в секрете", required=True, default="deploy-token")

        _binary = binary_task.binary_release_parameters_list(stable=True)

    @property
    def deploy_client(self):
        import metrika.pylib.deploy.client as deploy
        return deploy.DeployAPI(token=self.Parameters.tokens_secret.data().get(self.Parameters.deploy_token_key))

    def on_execute(self):
        self.run_subtasks([self.delete_stand(stand) for stand in self.list_stands() if self.is_forsaken(stand)])

    def is_forsaken(self, name):
        logging.info("Checking {name}".format(name=name))

        create_time = self.get_stand_create_time(name)
        logging.info("Creation time: {time}".format(time=create_time))

        age = (datetime.datetime.now() - create_time).days
        logging.info("age={age}".format(age=age))
        if age > self.Parameters.max_age:
            self.set_info("Стенд <b>{name}</b> будет удален: <b>устарел</b>, создан <b>{created}</b>, возраст <b>{age}</b> дней".format(name=name, created=create_time.strftime("%Y-%m-%d"), age=age),
                          do_escape=False)
            return True

        issue_key_match = re.search(r"-([a-z]+-\d+)", name)

        if issue_key_match:
            logging.info("Issue key match: {issue}".format(issue=issue_key_match.groups()))
            try:
                issue = self.st_client.issues[issue_key_match.group(1)]
                if issue.status.key == "closed":
                    self.set_info("Стенд <b>{name}</b> будет удален: <b>задача</b> {issue} <b>закрыта</b>".format(name=name, issue=issue.key), do_escape=False)
                    return True
            except Exception as e:
                logging.warning("No issue for key '{}'".format(issue_key_match.group(1)))
                self.set_info("Стенд <b>{name}</b> распарсился как <b>{key}</b>, Startrek ответил: '{error}'".format(name=name, key=issue_key_match.groups()[0], error=e), do_escape=False)
                return False

        return False

    def get_stand_create_time(self, stand):
        stand_history = self.deploy_client.select_object_history("stage", stand, selectors=["/user", "/time"])
        for obj in stand_history:
            if obj["user"] != "robot-drug-deploy":
                update_timestamp = obj["time"] // 10 ** 6
                break
        else:
            update_timestamp = self.deploy_client.stage.get_stage(stand, ["/status/validated/last_transition_time/seconds"])
        return datetime.datetime.fromtimestamp(update_timestamp)

    def list_stands(self):
        raise NotImplementedError()

    def delete_stand(self, stand):
        raise NotImplementedError()
