# coding=utf-8
import contextlib
import datetime
import hashlib
import logging

import requests

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common.decorators import memoized_property
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.admins.metrika_maas_update.state import Destination, Link, Source, State
from sandbox.projects.metrika.java.metrika_java_maas_parent_prepare import MetrikaJavaMaasParentPrepare
from sandbox.projects.metrika.utils import maas, settings
from sandbox.projects.metrika.utils.base_metrika_task import with_parents
from sandbox.projects.metrika.utils.maas import MaasProvider
from sandbox.projects.metrika.utils.mixins import juggler_reporter
from sandbox.projects.metrika.utils.parameters import DataCenterParameters
from sandbox.projects.metrika.utils.pipeline import pipeline
from sandbox.sdk2 import parameters


@with_parents
class MetrikaMaasUpdate(pipeline.PipelineBaseTask, juggler_reporter.JugglerReporterMixin):
    """
    Подвоз данных с тестинга MDB в MaaS
    """

    class Parameters(utils.CommonParameters):
        description = "Подвоз бекапов тестинга в MaaS"

        data_center_params = DataCenterParameters()

        is_use_backups = parameters.Bool("Использовать ли бекапы кластеров MDB", required=True, default_value=False,
                                         description="Если задано, то копирование будет производиться не из непосредственно кластеров тестинга, а из кластеров, поднятых из бекапов "
                                                     "указанных кластеров. Это потребует временно дополнительную квоту, примерно треть и дополнительное время на восстановление кластера, "
                                                     "однако процесс будет изолирован от воздействий на тестинг.")

        is_report_to_juggler = parameters.Bool("Отправлять ли событие в Juggler", required=True, default_value=False,
                                               description="Должно быть выставлено, если запуск производится регулярный, например, из шедулера.")

        with parameters.Group('bishop') as bishop_group:
            bishop_program = parameters.String("Программа", default="metrika-maas-update", required=True,
                                               description="Имя программы в bishop")
            bishop_environment = parameters.String("Окружение", default="metrika.sandbox.admin.production", required=True,
                                                   description="Имя окружения в bishop")

        with parameters.Group("Секреты") as secrets_group:
            bishop_token = parameters.YavSecretWithKey('Bishop token', required=True, default='{}#bishop_oauth_token'.format(settings.rma_yav_uuid),
                                                       description="Секрет с токеном для доступа к bishop")
            yav_token = parameters.YavSecretWithKey('Vault token', required=True, default='{}#vault_oauth_token'.format(settings.rma_yav_uuid),
                                                    description="Секрет с токеном для доступа к секретнице")
            yc_token = parameters.YavSecretWithKey('Cloud token', required=True, default='{}#ymdb_oauth_token'.format(settings.rma_yav_uuid),
                                                   description="Секрет с токеном для доступа облаку")

        _binary = binary_task.binary_release_parameters_list(stable=True)

    @property
    def juggler_host(self):
        return self.pipeline_state.maas_host

    def _juggler_predicate(self, status):
        return self.Parameters.is_report_to_juggler

    @property
    def pipeline_state(self):
        return State(self.Context.pipeline_state)

    class Context(pipeline.PipelineBaseTask.Context):
        pipeline_state = State().state

    def create_stages(self):
        stages = [
            (self.initialize, 'Подготовка'),
            (self.create_proto_root, 'Приёмник'),
        ]
        if self.Parameters.is_use_backups:
            stages.append((self.create_sources, 'Источники'))
        stages.extend(
            [
                (self.create_transfers, 'Трансферы'),
                (self.activate_transfers, 'Активация'),
                (self.wait_for_transfers, 'Ожидание трансферов'),
                (self.prepare_parents, 'Родительские инстансы'),
                (self.clean_up, 'Завершение'),
            ]
        )

        return stages

    @memoized_property
    def src_password(self):
        return self.vault_client.get_version(self.defaults.src.password.secret_id)["value"][self.defaults.src.password.key]

    @memoized_property
    def dst_password(self):
        return self.vault_client.get_version(self.defaults.dst.password.secret_id)["value"][self.defaults.dst.password.key]

    def mysql_connection(self, destination):
        import MySQLdb
        return contextlib.closing(
            MySQLdb.connect(
                host=destination.host,
                port=destination.port,
                user=self.defaults.dst.user,
                passwd=self.dst_password
            )
        )

    @memoized_property
    def mdb_mysql_client(self):
        import metrika.pylib.yc.mysql as mdb
        return mdb.ManagedMySQL(token=self.Parameters.yc_token.value())

    @memoized_property
    def data_transfer_client(self):
        import metrika.pylib.yc.data_transfer as data_transfer
        return data_transfer.DataTransfer(folder_id=self.config.folder_id, token=self.Parameters.yc_token.value())

    @memoized_property
    def vault_client(self):
        import metrika.pylib.vault as vault
        return vault.VaultClient(auth_type="oauth", oauth_token=self.Parameters.yav_token.value())

    @memoized_property
    def config(self):
        import metrika.pylib.structures.dotdict as dotdict
        return dotdict.DotDict.from_dict(self.pipeline_state.config)

    @memoized_property
    def destination_config(self):
        return self.config.maas[self.pipeline_state.maas_host]

    @memoized_property
    def defaults(self):
        return self.config.defaults

    @memoized_property
    def maas_provider(self):
        return MaasProvider(self.Parameters.bishop_token.value(), self.Parameters.data_center)

    def initialize(self):
        """
        Получить конфиг из бишопа
        :return:
        """

        self.pipeline_state.maas_host = self.maas_provider.get_maas_client().host

        import metrika.pylib.config as config

        self.pipeline_state.config = config.get_yaml_config_from_bishop(
            program=self.Parameters.bishop_program,
            environment=self.Parameters.bishop_environment,
            token=self.Parameters.bishop_token.value(),
            vault_client=self.vault_client)

        # обработаем конфиг и получим списки источников и приёмников
        self.pipeline_state.clear()
        for cluster_info in self.destination_config.src.clusters:
            source = Source()
            if 'name' in cluster_info:
                mysql_cluster = self.mdb_mysql_client.cluster_by_name(cluster_info.name, self.config.folder_id)
            else:
                mysql_cluster = self.mdb_mysql_client.cluster_by_id(cluster_info.id)
            source.original_id = mysql_cluster.id
            source.original_name = mysql_cluster.name
            if not self.Parameters.is_use_backups:
                source.id = mysql_cluster.id
                source.name = mysql_cluster.name
            source.database = cluster_info.src_database
            self.pipeline_state.add_source(source)

            destination = Destination()
            destination.host = self.pipeline_state.maas_host
            destination.database = cluster_info.dst_database
            self.pipeline_state.add_destination(destination)

            link = Link()
            link.destination_key = destination.key
            link.source_key = source.key
            self.pipeline_state.add_link(link)

    def create_proto_root(self):
        """
        Создать инстанс, который будет приёмником
        В нём должны быт созданы все БД
        :return:
        """
        with self.memoize_stage_global["src-maas-instance-{}".format(self.Parameters.data_center)](commit_on_entrance=False):
            # Находим преднастроенный инстанс с имененем Void-80
            void_id = self.maas_provider.get_maas_client().get_latest_instance_by_name("Void-80").id
            # Создаём в нём инстанс-приёмник
            instance = self.maas_provider.get_maas_client()
            instance.create(
                name="root-{}".format(self.id),
                version_tag=maas.VERSION_TAG,
                parent_id=void_id, ttl=int(datetime.timedelta(weeks=2).total_seconds())
            )
            self.pipeline_state.maas_instance_id = instance.id
            for destination in self.pipeline_state.destinations:
                destination.port = instance.ports["mysql"]
            self.set_info("Родительский инстанс создан: {} {}:{}".format(self.pipeline_state.maas_instance_id, instance.host, instance.ports["mysql"]))
        with self.memoize_stage_global["src-maas-databases-{}".format(self.pipeline_state.maas_host)](commit_on_entrance=False):
            for destination in self.pipeline_state.destinations:
                with self.mysql_connection(destination) as connection:
                    with contextlib.closing(connection.cursor()) as cursor:
                        cursor.execute("CREATE DATABASE {}".format(destination.database))

    def create_sources(self):
        """
        Создать кластера, которые будут источниками - из бекапов указанных кластеров

        !! Стадия находится в конвейере только если проставлена галка is_use_backups !!
        :return:
        """
        import udatetime
        # 1. Проверяем, что у каждого исходного кластера уже есть бекап
        # 2. фиксируем моменты времени у всех бекапов, берём самый поздний и проверяем, что он меньше чем utcnow - 5 минут, если нет - ждём 10 минут
        # 3. восстанавливаем из бекапов временные кластера
        # 4. Ждём операции восстановления

        with self.memoize_stage.all_backups_present(commit_on_entrance=False, commit_on_wait=False):
            # инвариант для выхода из мемойза - у каждого источника должен быть заполнен backup_id

            for source in self.pipeline_state.sources:
                mysql_cluster = self.mdb_mysql_client.cluster_by_id(source.original_id)
                backups = mysql_cluster.list_backups()
                if backups:
                    last_backup = max(backups, key=lambda b: b.started_at)
                    source.backup_id = last_backup.id
                else:
                    logging.warning("No backups for cluster {}.".format(source.original_id))
                    if source.backup_operation_id:
                        logging.debug("Backup operation exists with id {}".format(source.backup_operation_id))
                    elif source.is_parallel_backup_operation_detected:
                        logging.debug("Parallel backup operation was detected.")
                    else:
                        try:
                            backup_operation = mysql_cluster.backup()
                        except requests.HTTPError as e:
                            if e.response.status_code == 400 and e.response.json().get('code', None) == 9:
                                source.is_parallel_backup_operation_detected = True
                                logging.warning("Parallel backup operation was detected: {}".format(e.response.json().get('message', None)))
                            else:
                                raise
                        else:
                            source.backup_operation_id = backup_operation.id
                            logging.info("Initiate backup operation {}".format(source.backup_operation_id))

            sources_without_backups = [source for source in self.pipeline_state.sources if not source.backup_id]

            if sources_without_backups:
                # только если есть источники без бекапов - проверим операции
                operations = [self.mdb_mysql_client.get_operation(source.backup_operation_id) for source in self.pipeline_state.sources if source.backup_operation_id]
                logging.debug("Backup operations:\n" + "\n".join([str(operation) for operation in operations]))
                failed_operations = [operation for operation in operations if not operation.is_successful]
                if failed_operations:
                    # требует вмешательтва - можно вручную сделать бекапы и когда они будут созданы, то продолжить таску
                    raise Exception("\n".join([str(operation) for operation in failed_operations]))
                not_completed_operations = [operation for operation in operations if not operation.is_done]
                if not_completed_operations or any([source.is_parallel_backup_operation_detected for source in sources_without_backups]):
                    raise sdk2.WaitTime(time_to_wait=self.defaults.get("backup_poll_period", 60))
                else:
                    # операции все завершились, нужно снова войти сюда и пополнить список бекапов
                    raise sdk2.WaitTime(time_to_wait=1)

        with self.memoize_stage.backup_timestamp_check(commit_on_entrance=False, commit_on_wait=False):
            max_backup_time_stamps = max([self.mdb_mysql_client.get_backup(source.backup_id).started_at for source in self.pipeline_state.sources])
            logging.info("Max backup timestamp: {}".format(udatetime.to_string(max_backup_time_stamps)))

            # фиксируем момент времени, на который будет производиться восстановление бекапов кластеров
            target_time_stamp = udatetime.to_string(datetime.datetime.now(tz=udatetime.TZFixedOffset(0)).replace(second=0, microsecond=0) - datetime.timedelta(minutes=5))
            logging.info("Target timestamp: {}".format(target_time_stamp))

            if max_backup_time_stamps > (datetime.datetime.now(tz=udatetime.TZFixedOffset(0)) - datetime.timedelta(minutes=10)):
                # есть слишком свежий бекап, подождём
                logging.warning("Too young backup found, wait for some time")
                raise sdk2.WaitTime(time_to_wait=600)
            else:
                # бекапы достаточно настоялись, зафиксируем
                self.pipeline_state.target_timestamp = target_time_stamp

        for source in self.pipeline_state.sources:
            with self.memoize_stage_global["cluster-for-{}".format(source.original_id)](commit_on_entrance=False, commit_on_wait=False):
                mysql_cluster = self.mdb_mysql_client.cluster_by_id(source.original_id)
                last_backup = self.mdb_mysql_client.get_backup(source.backup_id)
                cluster_name = "maas-update-{}-{}".format(self.id, last_backup.source_cluster_id)
                source.name = cluster_name
                config_spec = {
                    "backupId": last_backup.id,
                    "name": cluster_name,
                    "time": self.pipeline_state.target_timestamp,
                    "description": "временный кластер восстановлен из бекапа кластера {} ({}) Sandbox-задачей {}".format(source.original_id, source.original_name, self.id),
                    "labels": {
                        "sandbox_task_id": str(self.id),
                        "maas": self.Parameters.data_center,
                        "environment": "development",
                    },
                    "environment": mysql_cluster.data.environment,
                    "configSpec": {
                        "version": "8.0",
                        "resources": {
                            "resourcePresetId": self.defaults.src.resourcePresetId,
                            "diskSize": mysql_cluster.data.config.resources.diskSize,
                            "diskTypeId": self.defaults.src.diskTypeId
                        }
                    },
                    "hostSpecs": [
                        {
                            "zoneId": self.defaults.src.dc
                        }
                    ],
                }
                operation = self.mdb_mysql_client.restore(config_spec)
                source.restore_operation_id = operation.id
                source.id = operation.metadata.clusterId
                self.set_info('Временный кластер <a href="https://yc.yandex-team.ru/folders/{folder}/managed-mysql/cluster/{id}">{title} ({original})</a>'
                              .format(folder=self.config.folder_id, id=source.id, title=source.name, original=mysql_cluster.name), do_escape=False)

        with self.memoize_stage_global.wait_for_operations(commit_on_entrance=False, commit_on_wait=False):
            logging.info("Start waiting for operations")
            operations = [self.mdb_mysql_client.get_operation(source.restore_operation_id) for source in self.pipeline_state.sources]
            logging.debug("\n".join([str(o) for o in operations]))

            operations_failed = [o for o in operations if not o.is_successful]
            if operations_failed:
                message = "\n".join([str(o) for o in operations_failed])
                logging.error("Following operations failed:\n{}".format(message))
                raise Exception(message)

            operations_not_complete = [o for o in operations if not o.is_done]
            if operations_not_complete:
                message = "\n".join([str(o) for o in operations_not_complete])
                logging.info("Following operations not completed:\n{}".format(message))
                raise sdk2.WaitTime(time_to_wait=self.defaults.restore_poll_period)
            else:
                logging.info("All operations completed.")

    def create_transfers(self):
        """
        1. Создать endpoint's иточников и приёмника
        2. Создать трансферы
        3. Активировать трансферы
        :return:
        """
        for source in self.pipeline_state.sources:
            with self.memoize_stage_global["src-endpoint-{}".format(source.id)](commit_on_entrance=False):
                src_config = {
                    'name': 'maas-{}-{}'.format(self.id, source.id),
                    'description': 'источник для синхронизации MaaS с тестингом {} ({})'.format(source.id, source.original_name),
                    'direction': 'SOURCE',
                    'settings': {
                        'mysql_source': {
                            'connection': {
                                'mdb_cluster_id': source.id,
                            },
                            'database': source.database,
                            'user': self.defaults.src.user,
                            'password': {
                                'raw': self.src_password
                            },
                            'exclude_tables': ['^tmp', '^old'],
                            "object_transfer_settings": {
                                "view": "AFTER_DATA",
                                "trigger": "AFTER_DATA"
                            }
                        }
                    }
                }
                src_endpoint = self.data_transfer_client.create_endpoint(src_config)
                source.endpoint_id = src_endpoint.id

        for destination in self.pipeline_state.destinations:
            with self.memoize_stage_global["dst-endpoint-{}-{}".format(destination.host, destination.database)](commit_on_entrance=False):
                # Name length must be between 2 and 63 characters long, name must contain only lowercase alphanumeric characters and hyphens, first character must be a lowercase letter and last
                # character must be an alphanumeric character.
                hash = hashlib.md5()
                hash.update(str(destination.host).encode('utf-8'))
                hash.update(str(destination.port).encode('utf-8'))
                hash.update(str(destination.database).encode('utf-8'))
                name = 'maas-{}-{}'.format(self.id, hash.hexdigest())
                dst_config = {
                    'name': name,
                    'description': 'приёмник в MaaS для синхронизации с тестингом {}:{} {}'.format(destination.host, destination.port, destination.database),
                    'direction': 'TARGET',
                    'settings': {
                        'mysql_target': {
                            'connection': {
                                'on_premise': {
                                    'host': destination.host,
                                    'port': str(destination.port)
                                }
                            },
                            'database': destination.database,
                            'user': self.defaults.dst.user,
                            'password': {
                                'raw': self.dst_password
                            },
                            'timezone': 'Europe/Moscow',
                            "object_transfer_settings": {
                                "view": "BEFORE_DATA",
                                "trigger": "AFTER_DATA"
                            }
                        }
                    }
                }
                dst_endpoint = self.data_transfer_client.create_endpoint(dst_config)
                destination.endpoint_id = dst_endpoint.id

        for link in self.pipeline_state.links:
            with self.memoize_stage_global["transfer-{}-{}".format(link.source_key, link.destination_key)](commit_on_entrance=False):
                transfer_config = {
                    'name': 'maas-{}-{}-{}'.format(self.id, self.pipeline_state.get_source_for_link(link).endpoint_id, self.pipeline_state.get_destination_for_link(link).endpoint_id),
                    'description': 'Синхронизация MaaS с тестингом {} -> {}'.format(link.source_key, link.destination_key),
                    'source_id': self.pipeline_state.get_source_for_link(link).endpoint_id,
                    'target_id': self.pipeline_state.get_destination_for_link(link).endpoint_id,
                    'type': 'SNAPSHOT_ONLY',
                    'runtime': {
                        'yt_runtime': {
                            'cpu': 0.2,
                            'ram': '4.0 GB'
                        }
                    }
                }
                transfer = self.data_transfer_client.create_transfer(transfer_config)
                link.transfer_id = transfer.id
                self.set_info('<a href="https://yc.yandex-team.ru/folders/{folder}/data-transfer/transfer/{id}">Трансфер кластера {title}</a>&nbsp;&mdash;&nbsp;<a href="{dashboard}">Дашборд</a>'
                              .format(folder=self.config.folder_id,
                                      id=link.transfer_id,
                                      title=self.pipeline_state.get_source_for_link(link).original_name,
                                      dashboard=transfer.dashboard.link), do_escape=False)

    def activate_transfers(self):
        """
        Активировать трансферы
        """
        for link in self.pipeline_state.links:
            with self.memoize_stage_global["activate-transfer-{}".format(link.transfer_id)](commit_on_entrance=False):
                transfer = self.data_transfer_client.get_transfer(link.transfer_id)
                operation = transfer.activate()
                link.activate_operation_id = operation.id

    def wait_for_transfers(self):
        """
        Ожидать завершения трансферов, если ошибка - падать так, что бы их можно было подопнуть и продолжить конвейер
        :return:
        """
        # ожидаем не операций, но трансферов, что бы трансфер можно было подопнуть, будет другая операция
        with self.memoize_stage.wait_for_transfers(commit_on_entrance=False, commit_on_wait=False):
            logging.info("Start waiting for transfers")

            transfers_not_completed = []
            transfers_errors = []
            for link in self.pipeline_state.links:
                transfer = self.data_transfer_client.get_transfer(link.transfer_id)
                logging.info("Checking transfer: {} {}".format(transfer.id, transfer.status))
                if transfer.status_in_progress:
                    transfers_not_completed.append(transfer)
                elif not transfer.status_is_success:
                    transfers_errors.append(transfer)

            if transfers_errors:
                message = "\n".join([str(t) for t in transfers_errors])
                logging.error("Following transfers failed:\n{}".format(message))
                raise Exception(message)

            if transfers_not_completed:
                message = "\n".join([str(t) for t in transfers_not_completed])
                logging.info("Following transfers not completed:\n{}".format(message))
                raise sdk2.WaitTime(time_to_wait=self.defaults.transfer_poll_period)
            else:
                logging.info("All transfers completed.")

    def clean_up(self):
        """
        1. Удалить трансферы
        2. Удалить эндпоинты
        3. Удалить кластеры-истоичники
        :return:
        """
        import requests
        for link in self.pipeline_state.links:
            with self.memoize_stage["delete-transfer-{}".format(link.transfer_id)](commit_on_entrance=False):
                try:
                    operation = self.data_transfer_client.get_transfer(link.transfer_id).delete()
                except requests.exceptions.HTTPError as e:
                    logging.exception("Exception in get_transfer")
                    if e.response.status_code == 404:
                        logging.warning("Transfer {} not found".format(link.transfer_id))
                    else:
                        raise
                else:
                    operation.wait_for_done()
                    operation.verify()

        for destination in self.pipeline_state.destinations:
            with self.memoize_stage["delete-dst-endpoint-{}".format(destination.endpoint_id)](commit_on_entrance=False):
                try:
                    operation = self.data_transfer_client.get_endpoint(destination.endpoint_id).delete()
                except requests.exceptions.HTTPError as e:
                    logging.exception("Exception in get_endpoint")
                    if e.response.status_code == 404:
                        logging.warning("Endpoint {} not found".format(destination.endpoint_id))
                    else:
                        raise
                else:
                    operation.wait_for_done()
                    operation.verify()

        for source in self.pipeline_state.sources:
            with self.memoize_stage["delete-src-endpoint-{}".format(source.endpoint_id)](commit_on_entrance=False):
                try:
                    operation = self.data_transfer_client.get_endpoint(source.endpoint_id).delete()
                except requests.exceptions.HTTPError as e:
                    logging.exception("Exception in get_endpoint")
                    if e.response.status_code == 404:
                        logging.warning("Endpoint {} not found".format(source.endpoint_id))
                    else:
                        raise
                else:
                    operation.wait_for_done()
                    operation.verify()

        if self.Parameters.is_use_backups:
            for source in self.pipeline_state.sources:
                with self.memoize_stage["delete-cluster-{}".format(source.id)](commit_on_entrance=False):
                    if source.id == source.original_id:
                        raise pipeline.PipelineAbortError("Не стоит удалять кластер-источник, если он не был восстановлен из бекапа. id={}, original_id={}".format(source.id, source.original_id))
                    try:
                        operation = self.mdb_mysql_client.cluster_by_id(source.id).delete()
                    except requests.exceptions.HTTPError as e:
                        logging.exception("Exception in cluster_by_id or delete operation")
                        if e.response.status_code == 403:
                            data = e.response.json()
                            if data.get('code') == 7:
                                logging.info(data.get('message'))
                                logging.warning("Cluster {} not found".format(source.id))
                            else:
                                raise
                        else:
                            raise
                    else:
                        operation.wait_for_done()
                        operation.verify()

    def prepare_parents(self):
        """
        Инициировать создание парента и ожидать завершения
        :return:
        """
        if self.destination_config.create_parents:
            parent_prepare = [
                (
                    MetrikaJavaMaasParentPrepare,
                    {
                        MetrikaJavaMaasParentPrepare.Parameters.parent_id.name: self.pipeline_state.maas_instance_id,
                        MetrikaJavaMaasParentPrepare.Parameters.data_center.name: self.Parameters.data_center
                    }
                )
            ]

            self.run_subtasks(parent_prepare)
        else:
            self.set_info("Создание родительских инстансов отключено")
