# coding=utf-8
import functools
import logging
import os
import re
import tempfile
import urllib2

from sandbox import sdk2
from sandbox.common.errors import TemporaryError
from sandbox.common.utils import progressive_waiter
from sandbox.projects.common import binary_task
from sandbox.projects.common.debpkg import DebRelease
from sandbox.projects.common.decorators import retries
from sandbox.projects.common.gnupg import GpgKey2
from sandbox.projects.metrika.utils import CommonParameters
from sandbox.projects.metrika.utils import parameters
from sandbox.projects.metrika.utils.base_metrika_task import with_parents, BaseMetrikaTask
from sandbox.projects.metrika.utils.mixins.console import BaseConsoleMixin
from sandbox.sandboxsdk.paths import add_write_permissions_for_path, copy_path
from sandbox.sdk2.ssh import Key

PUBLIC_REPO_HOST = "repo.mirror.yandex.net"


def _get_packages(changes_file):
    with open(changes_file, 'r') as changes_file:
        for line in changes_file.readlines():
            if line.startswith("Binary: "):
                return line[8:].split(' ')


def _get_version(changes_file):
    with open(changes_file, 'r') as changes_file:
        for line in changes_file.readlines():
            if line.startswith("Version: "):
                return line[9:]


def _get_only_binary_changes_files_path(artifacts):
    changes_files = _get_changes_files_path(artifacts)
    return filter(lambda f: not f.endswith('source.changes'), changes_files)


def _get_changes_files_path(artifacts):
    changes_files = filter(lambda f: f.endswith('.changes'), os.listdir(artifacts))
    logging.debug("Changes files: %s", changes_files)
    return [os.path.join(artifacts, changes_file) for changes_file in changes_files]


def _collect_packages_for_wait(changes_files):
    expected_packages = {}
    for changes_file in changes_files:
        version = _get_version(changes_file)
        for package in _get_packages(changes_file):
            expected_packages[package.strip()] = version.strip()
    return expected_packages


class DuploadParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group("dupload") as sign_options_block:
        login = sdk2.parameters.String("Логин", required=True, default="robot-metrika-test")
        ssh_private_key_vault_name = sdk2.parameters.String("SSH Private Key Vault Name",
                                                            required=True,
                                                            default="robot-metrika-test-ssh",
                                                            description="Имя секрета в котором находится приватный "
                                                                        "SSH ключ для заливки пакетов")
        ssh_private_key_vault_owner = sdk2.parameters.String("SSH Private Key Vault Owner",
                                                             required=True,
                                                             default="METRIKA",
                                                             description="Имя владельца секрета в котором находится "
                                                                         "приватный SSH ключ для заливки пакетов")

        gpg_private_key_vault_name = sdk2.parameters.String("secring.gpg Vault Name",
                                                            required=True,
                                                            default="robot-metrika-test-gpg-private",
                                                            description="Имя секрета в котором находится приватный "
                                                                        "GPG ключ")
        gpg_public_key_vault_name = sdk2.parameters.String("pubring.gpg Vault Name",
                                                           required=True,
                                                           default="robot-metrika-test-gpg-public",
                                                           description="Имя секрета в котором находится публичный "
                                                                       "GPG ключ")
        gpg_key_vault_owner = sdk2.parameters.String("GPG secring and pubring Vault Owner",
                                                     required=True,
                                                     default="METRIKA",
                                                     description="Имя владельца секрета в котором находится "
                                                                 "приватный и публичный GPG ключи")
        gpg_passphrase_vault_name = sdk2.parameters.String("GPG Passphrase Vault Name",
                                                           required=True,
                                                           default="robot-metrika-test-gpg-passphrase",
                                                           description="Имя секрета в котором находится парольная"
                                                                       "фраза для GPG")
        gpg_passphrase_vault_owner = sdk2.parameters.String("GPG Passphrase Vault Owner",
                                                            required=True,
                                                            default="METRIKA",
                                                            description="Имя владельца секрета в котором находится "
                                                                        "парольная фраза для GPG")


@with_parents
class MetrikaPackagesUpload(BaseMetrikaTask, BaseConsoleMixin):
    """
    Загружает пакеты  в репозиторий
    """

    class Parameters(CommonParameters):
        max_restarts = 20
        container = parameters.LastPeasantContainerResource("Environment container resource", required=True)
        artifacts = sdk2.parameters.Resource("Артефакты",
                                             required=True,
                                             description="Ресурс, в котором располагаются артефакты для загрузки")
        directory = sdk2.parameters.String("Каталог для загрузки",
                                           required=True,
                                           default='.',
                                           description="Относительный каталог в ресурсе, который содержит файлы для "
                                                       "загрузки")
        repository = sdk2.parameters.String("Репозиторий",
                                            required=True,
                                            description="Репозиторий для загрузки")
        wait_for_dist = sdk2.parameters.Bool("Ожидать пакетов на dist", default=True)

        dupload_block = DuploadParameters()

        with wait_for_dist.value[True]:
            with sdk2.parameters.Group("Wait") as wait:
                branches = sdk2.parameters.List("Ветка",
                                                required=True,
                                                default=['unstable', 'testing', 'prestable', 'stable'],
                                                description="Ветки в одной из которых ожидается появление пакета")
                arches = sdk2.parameters.List("Архитектура",
                                              required=True,
                                              default=['all', 'amd64', 'i386'])

                with sdk2.parameters.Group("Wait options") as wait_options_block:
                    tick = sdk2.parameters.Integer("tick",
                                                   required=True,
                                                   default=30,
                                                   description="Initial tick amount in seconds.")
                    max_tick = sdk2.parameters.Integer("max_tick",
                                                       required=True,
                                                       default=120,
                                                       description="Maximum available tick amount in seconds.")
                    max_wait = sdk2.parameters.Integer("max_wait",
                                                       required=True,
                                                       default=1800,
                                                       description="Maximum available tick amount in seconds.")

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def on_execute(self):
        artifacts = os.path.join(self._retrieve_artifacts(), self.Parameters.directory)
        with self.memoize_stage.upload(commit_on_entrance=False):
            self._upload_to_dist(artifacts)
        if self.Parameters.wait_for_dist:
            with self.memoize_stage.wait(commit_on_entrance=False):
                self._wait_for_packages(artifacts)

    def _retrieve_artifacts(self):
        with sdk2.helpers.ProgressMeter("Retrieving artifacts"):
            ro_path = str(sdk2.ResourceData(self.Parameters.artifacts).path)
            rw_path = str(self.path('artifacts'))
            copy_path(ro_path, rw_path)
            add_write_permissions_for_path(rw_path)
            return rw_path

    @retries(5)
    def _upload_to_dist(self, root_dir):
        with sdk2.helpers.ProgressMeter("Upload to dist"):
            dupload_config = {
                self.Parameters.repository: {
                    'dinstall_runs': 0
                }
            }
            if self._is_on_build_farm():
                incoming_base_dir = '/home/buildfarm/incoming/clickhouse'
                dupload_config[self.Parameters.repository]['fqdn'] = PUBLIC_REPO_HOST
                dupload_config[self.Parameters.repository]['incoming'] = incoming_base_dir + '/deb/' + self._build_farm_type()
                dupload_config[self.Parameters.repository]['login'] = 'buildfarm'
            elif self._is_metrika_precise():
                dupload_config[self.Parameters.repository]['fqdn'] = 'metrika.dupload.dist.yandex.ru'
                dupload_config[self.Parameters.repository]['incoming'] = '/repo/metrika/mini-dinstall/incoming/'

            with DebRelease(dupload_config, self.Parameters.login):
                with GpgKey2(self.Parameters.gpg_key_vault_owner,
                             self.Parameters.gpg_private_key_vault_name,
                             self.Parameters.gpg_public_key_vault_name):
                    with Key(self, self.Parameters.ssh_private_key_vault_owner,
                             self.Parameters.ssh_private_key_vault_name):
                        pass_phrase = sdk2.Vault.data(self.Parameters.gpg_passphrase_vault_owner,
                                                      self.Parameters.gpg_passphrase_vault_name)

                        with tempfile.NamedTemporaryFile() as pass_phrase_file:
                            pass_phrase_file.write(pass_phrase)
                            pass_phrase_file.flush()

                            self._execute_script('''\
set -o errexit
set -o xtrace

find {root_dir}
gpg --list-keys
chmod --recursive +w {root_dir}
debsign -k{login} -p"gpg --verbose --no-use-agent --batch --no-tty --passphrase-file {pass_phrase_file}" {changes}
find {root_dir} -type f -print0 | xargs -0 -I{{}} chmod 0644 {{}}
ls -la {root_dir}
dupload --nomail --to {repo} {root_dir}
                            '''.format(root_dir=root_dir, login=self.Parameters.login,
                                       pass_phrase_file=pass_phrase_file.name,
                                       changes=" ".join(_get_changes_files_path(root_dir)),
                                       repo=self.Parameters.repository))

                            # специальный случай публикации пакетов ClickHouse
                            if self._is_on_build_farm():
                                env = os.environ.copy()
                                env['SUBDIR'] = self._build_farm_type()
                                env['USERNAME'] = self.Parameters.login
                                env['HOST'] = dupload_config[self.Parameters.repository]['fqdn']
                                env['DIR'] = incoming_base_dir
                                self._execute_script('''\
set -o errexit
set -o xtrace

fakeroot alien --to-rpm --scripts clickhouse*.deb
scp *.rpm ${USERNAME}@${HOST}:${DIR}/rpm/${SUBDIR}/

if [[ "$SUBDIR" == "stable" ]];
then
    fakeroot alien --to-tgz --scripts clickhouse*.deb
    scp *.tgz ${USERNAME}@${HOST}:${DIR}/tgz/
fi
                                ''', cwd=root_dir, env=env)

    def _is_on_build_farm(self):
        return str(self.Parameters.repository).startswith('buildfarm')

    def _build_farm_type(self):
        match = re.search('^buildfarm-(?P<type>stable|testing)$', self.Parameters.repository)
        return match.group('type') if match else None

    def _is_metrika_precise(self):
        return self.Parameters.repository == 'metrika-precise'

    def _wait_for_packages(self, artifacts):
        with sdk2.helpers.ProgressMeter("Wait for packages"):
            expected_packages = _collect_packages_for_wait(_get_only_binary_changes_files_path(artifacts))
            logging.info("Expected packages: %s", expected_packages)

            if not progressive_waiter(self.Parameters.tick, self.Parameters.max_tick, self.Parameters.max_wait,
                                      functools.partial(self._check_repo, expected_packages=expected_packages))[0]:
                raise TemporaryError("Not all packages available in repository")

    def _check_repo(self, expected_packages):
        """
        Скачивает все Packages-файлы и ищет в них заданные пакеты

        :return: true если в Packages файлах находятся все пакеты и их реально можно скачать
        """
        try:
            packages_files = []
            for branch in self.Parameters.branches:
                for arch in self.Parameters.arches:
                    packages_files.append(
                        "http://dist.yandex.ru/{}/{}/{}/Packages".format(self.Parameters.repository if not self._is_metrika_precise() else "metrika", branch, arch))

            parsed_packages_files = [self._parse_package_file(package_file) for package_file in packages_files]

            logging.info("Expected packages: %s", expected_packages)

            state = {package: any([
                any([
                    p['Package'] == package and p['Version'] == version for p in parsed_packages_file
                ]) for parsed_packages_file in parsed_packages_files
            ]) for package, version in expected_packages.iteritems()}

            logging.info("State: %s", state)

            return all(state.values())

        except:
            logging.warning("Exception in _check_repo continue", exc_info=True)
            return False

    @staticmethod
    def _parse_package_file(package_file):
        """
        Скачивает Packages-файл и разбирает его на записи
        :param package_file: url
        :return: список словарей
        """
        result = []
        logging.debug("Parse %s", package_file)
        try:
            for chunk in urllib2.urlopen(package_file).read().split('\n\n'):
                d = dict()
                for line in chunk.splitlines():
                    split = line.split(": ", 2)
                    if len(split) > 1:
                        k = split[0]
                        v = split[1]
                        if k == 'Package' or k == 'Version':
                            d[k] = v if v else None
                if d:
                    result.append(d)
        except:
            logging.warning("Exception in parsing %s. Ignore.", package_file, exc_info=True)

        return result
