# coding=utf-8
import logging
import re
import yaml

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.metrika.utils.base_metrika_task import with_parents, BaseMetrikaTask
from sandbox.common.utils import progressive_waiter
from sandbox.sdk2.vcs.svn import Arcadia, SvnError
from sandbox.common import errors, patterns
from sandbox.projects.metrika.utils import arcanum_api


@with_parents
class MetrikaUpdateCiRegistry(BaseMetrikaTask):
    """
    Устанавливает заданную stable-версию у CI-тасклетов.
    """

    class Parameters(sdk2.Parameters):
        ci_tasklets = sdk2.parameters.List("CI tasklet ids", required=True,
                                           description="Список идентификаторов CI-тасклетов")
        resource = sdk2.parameters.Resource("Tasklet Binary Resource", required=True,
                                            description="Ресурс с загруженным тасклетом")
        wait_for_merged = sdk2.parameters.Bool("Ожидать мержа ревью", default_value=True)
        with wait_for_merged.value[True]:
            with sdk2.parameters.Group("Wait options") as wait_options_block:
                tick = sdk2.parameters.Integer("tick",
                                               required=True,
                                               default=30,
                                               description="Initial tick amount in seconds.")
                max_tick = sdk2.parameters.Integer("max_tick",
                                                   required=True,
                                                   default=120,
                                                   description="Maximum available tick amount in seconds.")
                max_wait = sdk2.parameters.Integer("max_wait",
                                                   required=True,
                                                   default=1800,
                                                   description="Maximum available tick amount in seconds.")

        login = sdk2.parameters.String("SSH User", required=True, default_value="robot-metrika-test")
        ssh_key = sdk2.parameters.Vault("SSH key in Vault", required=True, default_value="METRIKA:robot-metrika-test-ssh",
                                        description="'name' or 'owner:name'")
        arcanum_token = sdk2.parameters.Vault("Arcanum OAuth token in vault", required=True, default_value="METRIKA:robot-metrika-test-arcanum-token",
                                              description="'name' or 'owner:name'")

        _binary = binary_task.binary_release_parameters_list(stable=True)

        with sdk2.parameters.Output:
            review_id = sdk2.parameters.Integer("Review ID")

    @patterns.singleton_property
    def arcanum_api(self):
        return arcanum_api.ArcanumApi(token=self.Parameters.arcanum_token.data())

    @patterns.singleton_property
    def wd(self):
        path = self.path('wd')
        path.mkdir()
        return path

    def on_execute(self):
        with self.memoize_stage.review(commit_on_entrance=False):
            self._create_review()
        if self.Parameters.wait_for_merged:
            with self.memoize_stage.merged(commit_on_entrance=False):
                self._wait_for_merged()

    def _update_registry(self):
        for ci_tasklet in self.Parameters.ci_tasklets:
            yaml_path = (self.wd / ci_tasklet).with_suffix(".yaml")
            logging.info("Processing: {}".format(yaml_path.as_posix()))
            content = yaml.safe_load(yaml_path.read_text(encoding="utf-8"))
            content["versions"]["stable"] = self.Parameters.resource.id
            yaml_path.write_bytes(yaml.safe_dump(content, default_flow_style=False, allow_unicode=True).encode("utf-8"))

    def _create_review(self):
        with sdk2.helpers.ProgressMeter("Updating resource ids"):

            Arcadia.checkout('arcadia:/arc/trunk/arcadia/ci/registry', self.wd.as_posix())

            self._update_registry()

            with sdk2.ssh.Key(self, self.Parameters.ssh_key.owner, self.Parameters.ssh_key.name):
                try:
                    commit_result = Arcadia.commit(self.wd.as_posix(), 'Update CI Registry with resource {}'.format(self.Parameters.resource.id), user=self.Parameters.login)
                    logging.debug("Got commit_result while creating review: %s", commit_result)
                except SvnError as e:
                    logging.warning("Exception in commit.", exc_info=True)
                    if e.error_code != 'E165001':  # blocked by pre-commit hook
                        raise
                    else:
                        created_review = re.findall(
                            r"Check status can be monitored using this special review request: ([0-9]+)",
                            str(e),
                        )
                        if created_review:
                            created_review = created_review[0]
                            logging.info("Created review %s", created_review)
                            self.set_info('<a href="https://a.yandex-team.ru/review/{id}">Ревью {id}</a>'.format(id=created_review), do_escape=False)
                            self.Parameters.review_id = int(created_review)
                        else:
                            raise errors.TaskError("Review was not created")

    def _wait_for_merged(self):
        with sdk2.helpers.ProgressMeter("Wait for review merged"):
            if not progressive_waiter(self.Parameters.tick, self.Parameters.max_tick, self.Parameters.max_wait,
                                      self._check_review_merged)[0]:
                raise errors.TaskError("Review {} was not merged".format(self.Parameters.review_id))

    def _check_review_merged(self):
        review_status = self.arcanum_api.get_review_request(self.Parameters.review_id).json()["status"]
        if review_status not in {"pending_review", "submitted"}:
            logging.error("Unexpected review status {}".format(review_status))
            raise errors.TaskError("Unexpected review status {}".format(review_status))
        elif review_status == "submitted":
            committed_revision = self.arcanum_api.get_review_request(self.Parameters.review_id).json()["commits"][0]["committedAtRevision"]
            logging.info("Review merged. Revision: {}".format(committed_revision))
            self.set_info('<a href="https://a.yandex-team.ru/arc/commit/{}">Коммит</a>'.format(committed_revision), do_escape=False)
            return True
        else:
            return False
