# coding=utf-8
import time
import yaml

import sandbox.common.errors as errors
import sandbox.sdk2 as sdk2
import sandbox.projects.common.binary_task as binary_task

import sandbox.projects.metrika.utils as utils
import sandbox.projects.metrika.utils.base_metrika_task as base_metrika_task

import sandbox.projects.metrika.utils.settings as settings


@base_metrika_task.with_parents
class MetrikaYandexTankStandCreate(binary_task.LastRefreshableBinary, sdk2.Task):
    class Parameters(utils.CommonParameters):
        with sdk2.parameters.Group('Stage parameters') as stage_parameters:
            stage_id = sdk2.parameters.String(
                'Stage ID',
                required=True,
            )
            stage_image_name = sdk2.parameters.String(
                'Image name',
                required=True,
                default='load/yandex-tank-internal',
            )
            stage_image_tag = sdk2.parameters.String(
                'Image tag',
                required=True,
                default='1.13.7-artifacts-rc12',
            )
            with sdk2.parameters.RadioGroup('Stage datacenter') as stage_dc:
                stage_dc.values['vla'] = stage_dc.Value(default=True)
                stage_dc.values['sas'] = None
                stage_dc.values['iva'] = None

            stage_project_name = sdk2.parameters.String(
                'Project name',
                required=True,
                default='metrika-load-testing',
            )
            stage_endpoint_port = sdk2.parameters.Integer(
                'Endpoint port',
                required=True,
                default=8083,
            )
            stage_vcpu_limit = sdk2.parameters.Integer(
                'VCPU limit',
                required=True,
                default=16000,
            )
            stage_memory_limit = sdk2.parameters.Integer(
                'Memory limit',
                required=True,
                default=17179869184,
            )
            stage_disk_capacity = sdk2.parameters.Integer(
                'Disk capacity',
                required=True,
                default=161061273600,
            )
            stage_network_macro = sdk2.parameters.String(
                'Network macro',
                required=True,
                default='_YMETRICTESTNETS_',
            )
            stage_abc_service_id = sdk2.parameters.Integer(
                'ABC Service id',
                required=True,
                default=30740,  # metrika-ci
            )
        with sdk2.parameters.Group('Secrets parameters') as secrets_parameters:
            deploy_token_yav_secret = sdk2.parameters.YavSecret(
                'YP access secret',
                required=True,
                default=settings.yav_uuid,
                description='Used in requests to yp api',
            )
            deploy_token_yav_secret_key = sdk2.parameters.String(
                'YP access secret key',
                required=True,
                default='deploy-token',
            )
        with sdk2.parameters.Output:
            output_host = sdk2.parameters.String("Host")
            output_port = sdk2.parameters.Integer("Port")

        _binary = binary_task.binary_release_parameters_list(stable=True)

    class Context(sdk2.Task.Context):
        spec = None

    @property
    def deploy_client(self):
        import metrika.pylib.deploy.client

        secret = self.Parameters.deploy_token_yav_secret.data()
        token = secret[self.Parameters.deploy_token_yav_secret_key]

        return metrika.pylib.deploy.client.DeployAPI(token=token)

    @property
    def endpoint_set_id(self):
        return '%s.Tank' % self.Parameters.stage_id

    def on_execute(self):
        spec_text = utils.render(
            'resources/tank-stage-spec.j2',
            context={
                'stage_id': self.Parameters.stage_id,
                'stage_image_name': self.Parameters.stage_image_name,
                'stage_image_tag': self.Parameters.stage_image_tag,
                'stage_dc': self.Parameters.stage_dc,
                'stage_project_name': self.Parameters.stage_project_name,
                'stage_endpoint_port': self.Parameters.stage_endpoint_port,
                'stage_abc_service_id': self.Parameters.stage_abc_service_id,
                'stage_disk_capacity': self.Parameters.stage_disk_capacity,
                'stage_network_macro': self.Parameters.stage_network_macro,
                'stage_vcpu_limit': self.Parameters.stage_vcpu_limit,
                'stage_memory_limit': self.Parameters.stage_memory_limit,
            },
        )
        self.Context.spec = spec_text
        spec = yaml.safe_load(spec_text)

        self.deploy_client.stage.deploy_stage(
            spec,
            remove_if_exists=True,
            wait=True,
            wait_kwargs={'timeout': 60 * 30},
        )

        host, port = self.get_stage_address()

        self.Parameters.output_host = host
        self.Parameters.output_port = port

    def get_stage_address(self):
        import metrika.pylib.deploy.resolver as mpdr
        resolver = mpdr.Resolver(
            client_name=self.__class__.__name__,
        )

        timeout = 60 * 5
        poll_period = 15

        host = None
        port = None

        start_ts = time.time()
        current_ts = time.time()

        while True:
            if (current_ts - start_ts) > timeout:
                raise errors.TaskError('Failed to resolve stage endpoints: timeout reached ({}s)'.format(timeout))

            endpoints = resolver.resolve_endpoints(
                endpoint_set_id=self.endpoint_set_id,
                datacenter=self.Parameters.stage_dc,
            )
            if endpoints:
                host = endpoints[0].fqdn
                port = endpoints[0].port
                break

            time.sleep(poll_period)
            current_ts = time.time()

        return host, port
