# coding=utf-8
from __future__ import unicode_literals

import logging
import os
import re

import jinja2

import json
from sandbox.projects.common import binary_task
from sandbox.projects.metrika.utils import CommonParameters
from sandbox.projects.metrika.utils import settings, vcs
from sandbox.projects.metrika.utils.base_metrika_task import with_parents, BaseMetrikaTask
from sandbox.projects.metrika.utils.mixins.juggler_reporter import JugglerReporterMixin
from sandbox.sdk2 import ssh
from sandbox.sdk2.vcs.svn import Arcadia, SvnError

PACKAGE_JSON = 'package.json'
DOCKER_JSON = 'docker.json'
TARBALL_JSON = 'tarball.json'
CONFIG_JSON = 'config-tarball.json'
CONFIG_FILES = [PACKAGE_JSON, DOCKER_JSON, TARBALL_JSON, CONFIG_JSON]


TEMPLATE = """\
# autogenerated in sandbox/projects/metrika/utils/programs_list
{%- for group, programs in all_programs | dictsort %}
{{ group | upper }}_PROGRAMS = [
  {%- for program in programs | sort %}
    '{{ program }}',
  {%- endfor %}
]
{% endfor -%}

"""


@with_parents
class MetrikaProgramsList(BaseMetrikaTask, JugglerReporterMixin):
    juggler_host = 'metrika-sandbox'

    class Parameters(CommonParameters):
        _binary = binary_task.binary_release_parameters_list(stable=True)

    @staticmethod
    def get_all_programs(arcadia_url, path, replace_suffix=True):
        programs = {}
        with vcs.mount_arc(arcadia_url) as arcadia:
            for directory, _, files in os.walk(os.path.join(arcadia, path)):
                for config in CONFIG_FILES:
                    if config in files:
                        logging.debug('%s/%s', directory, config)
                        with open(os.path.join(directory, config)) as config_json_file:
                            config_json = json.load(config_json_file)
                        if 'meta' not in config_json:
                            continue

                        program_name = config_json["meta"]["name"]
                        if replace_suffix:
                            program_name = program_name.replace("-metrika-yandex", "").replace("-yandex", "")
                        config_path = os.path.join(os.path.relpath(directory, arcadia), config)
                        programs.get(program_name).update({config: config_path}) if program_name in programs else programs.update({program_name: {config: config_path}})
        return programs

    @staticmethod
    def save_text_to_utils(task, text, rel_path_from_utils):
        path = task.path('wd')
        path.mkdir()

        Arcadia.checkout('arcadia:/arc/trunk/arcadia/sandbox/projects/metrika/utils', path.as_posix())
        path = path.joinpath(rel_path_from_utils)
        if not path.exists():
            raise Exception('%s must exist and be included in ya.make', path)
        path.write_text(text, encoding='utf8')
        try:
            Arcadia.add(path.as_posix(), parents=True)
        except SvnError as e:
            if e.error_code != 'E200009':  # already versioned
                raise

        with ssh.Key(task, settings.owner, settings.ssh_key):
            try:
                Arcadia.commit(path.parent.as_posix(), 'Update {}'.format(rel_path_from_utils), user=settings.login)
            except SvnError as e:
                if e.error_code != 'E165001':  # blocked by pre-commit hook
                    raise
                else:
                    task.set_info('<a href="{}">Ревью</a>'.format(re.search(r'https://a\.yandex-team\.ru.*', str(e)).group()), do_escape=False)

    def on_execute(self):
        all_programs = {
            dir: self.get_all_programs('arcadia-arc:/#trunk', 'metrika/' + dir, replace_suffix=False)
            for dir in ['admin', 'java', 'core']
        }
        text = jinja2.Environment(loader=jinja2.BaseLoader).from_string(TEMPLATE).render({'all_programs': all_programs})

        self.save_text_to_utils(self, text, 'programs/__init__.py')
