# coding=utf-8
from __future__ import unicode_literals

import os
import shlex

import sandbox.common.types.resource as ctr
from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.common import binary_task, decorators
from sandbox.projects.metrika.utils import CommonParameters
from sandbox.projects.metrika.utils.base_metrika_task import with_parents, BaseMetrikaTask
from sandbox.projects.metrika.utils.task_types.ya_make import MetrikaYaMake
from sandbox.projects.metrika.utils.mixins.juggler_reporter import JugglerReporterMixin
from sandbox.projects.metrika.utils.resource_types import BaseMetrikaBinaryResource
from sandbox.sdk2.helpers import gdb


@with_parents
class RunBinary(BaseMetrikaTask, JugglerReporterMixin):
    class Parameters(CommonParameters):
        with sdk2.parameters.Group('Бинарник') as binary:
            binary_from_resource = sdk2.parameters.Bool('Бинарник из ресурса', default=True)
            with binary_from_resource.value[True]:
                binary_by_id = sdk2.parameters.Bool(
                    'Вычислить по id',
                    default=True,
                    description='Либо указать тип и атрибуты',
                )
                with binary_by_id.value[True]:
                    resource = sdk2.parameters.Resource('Ресурс', required=True)
                with binary_by_id.value[False]:
                    resource_type = sdk2.parameters.String('Тип ресурса', required=True)
                    resource_attrs = sdk2.parameters.Dict('Атрибуты ресурса')

            with binary_from_resource.value[False]:
                arcadia_url = sdk2.parameters.ArcadiaUrl(
                    'URL Аркадии',
                    required=True,
                    default_value='arcadia-arc:/#trunk',
                )
                yamake_path = sdk2.parameters.String(
                    'Путь до yamake',
                    required=True,
                    description='Например, relative/path/from/arcadia/leet_project (дефолтная релиз сборка)',
                )

        with sdk2.parameters.Group('Запуск') as run:
            argv = sdk2.parameters.String('Строка аргументов', description='-kek --lul 1337')
            env_vars = sdk2.parameters.Dict('Переменные окружения', description='key value')
            retries = sdk2.parameters.Integer('Количество дополнительных попыток', default=0)
            save_stdout = sdk2.parameters.Bool('Сохранить stdout в переменную контекста', default=False)

        with sdk2.parameters.Group('Juggler') as juggler:
            send_to_juggler = sdk2.parameters.Bool('Отправить результат в Juggler', default=False)
            with send_to_juggler.value[True]:
                event_from_stdout = sdk2.parameters.Bool(
                    'Событие из stdout программы',
                    default=True,
                    required=True,
                    description=(
                        'Событие парсится из последней строки stdout вида '
                        '[status description] или [service host status description]'
                    ),
                )
                with event_from_stdout.value[True]:
                    raise_if_crit = sdk2.parameters.Bool('Бросить исключение, если CRIT', default=True)

                juggler_service = sdk2.parameters.String('Сервис', description='Переопределяет stdout')
                juggler_host = sdk2.parameters.String('Хост', description='Переопределяет stdout')

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def on_prepare(self):
        self.juggler_host = self.Parameters.juggler_host
        self.juggler_service = self.Parameters.juggler_service

    def on_execute(self):
        env = {
            # for binary
            'ROBOT_METRIKA_ADMIN_OAUTH': sdk2.yav.Secret('sec-01cq6h07rwpqmqzb15y08jbs5q').data()['admin_oauth_token']
        }
        for k, v in self.Parameters.env_vars.items():
            if v.startswith('sec-'):
                ver, var = v.split(':', 1)
                v = sdk2.yav.Secret(ver).data()[var]
            env[k] = v

        if self.Parameters.binary_from_resource:
            if self.Parameters.binary_by_id:
                binary = self.Parameters.resource
            else:
                binary = sdk2.Resource.find(
                    type=self.Parameters.resource_type,
                    state=ctr.State.READY,
                    attrs=self.Parameters.resource_attrs
                ).first()
                self.set_info(
                    '<a href="https://sandbox.yandex-team.ru/resource/{}">Resource</a>'.format(binary.id),
                    do_escape=False,
                )
            binary_path = sdk2.ResourceData(binary).path.as_posix()
        else:
            self.Context.yamake_task = self.run_subtasks([(
                MetrikaYaMake,
                dict(
                    checkout_arcadia_from_url=self.Parameters.arcadia_url,
                    targets=self.Parameters.yamake_path,
                    resource_type=BaseMetrikaBinaryResource
                )
            )])[0]

            binary = sdk2.ResourceData(
                sdk2.Resource.find(
                    task_id=self.Context.yamake_task,
                    state=ctr.State.READY,
                    type=resource_types.BUILD_OUTPUT
                ).first()
            )
            binary_dir = binary.path.joinpath(self.Parameters.yamake_path)
            binary_path = next((
                path.as_posix() for path in binary_dir.glob('*')
                if path.is_file() and os.access(path.as_posix(), os.X_OK) and not path.suffix
            ), None)
        if not binary_path:
            raise Exception('Binary not found')

        with sdk2.helpers.ProcessLog(self, logger='run-binary') as log:
            for stream in ['out', 'err']:
                self.set_info(gdb.get_html_view_for_logs_file(
                    'Std' + stream, getattr(log, 'std' + stream).path.relative_to(self.log_path()), self.log_resource),
                    do_escape=False
                )

            decorators.retries(max_tries=self.Parameters.retries)(sdk2.helpers.subprocess.check_call)(
                [binary_path] + shlex.split(self.Parameters.argv),
                env=env, stdout=log.stdout, stderr=log.stderr
            )

            if self.Parameters.event_from_stdout:
                self.Context.stdout_path = log.stdout.path.as_posix()

            if self.Parameters.save_stdout:
                self.Context.stdout = log.stdout.path.read_text(encoding='utf8')

    def get_event(self):
        if self.Parameters.event_from_stdout and self.Context.stdout_path:
            service = host = None
            info = open(self.Context.stdout_path).readlines()[-1].split(' ', 3)
            if len(info) == 2:
                status, description = info
            elif len(info) == 4:
                service, host, status, description = info
            else:
                status, description = (
                    'CRIT',
                    'Program must print [status description] or [service host status description] line',
                )
            self.Context.status = status

            return self.juggler_service or service, self.juggler_host or host, status, description.strip()
        else:
            return super(RunBinary, self).get_event()

    def _send_juggler_event(self, status):
        if self.Parameters.send_to_juggler:
            super(RunBinary, self)._send_juggler_event(status)

    def on_finish(self, prev_status, status):
        if (
            self.Parameters.send_to_juggler and
            self.Parameters.event_from_stdout and
            self.Parameters.raise_if_crit and
            self.Context.status.lower() == 'crit'
        ):
            raise Exception('Program exited with CRIT status, see logs for details')
