# coding=utf-8
from __future__ import unicode_literals

import ast
import json
import logging
import os
import re
from collections import defaultdict
from pathlib2 import Path

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.format import ident
from sandbox.projects.common import binary_task
from sandbox.projects.common.binary_task import LastBinaryTaskRelease
from sandbox.projects.common.binary_task import LastRefreshableBinary
from sandbox.projects.metrika.utils import CommonParameters, vcs, render, settings, arcanum_api
from sandbox.projects.metrika.utils.base_metrika_task import with_parents, BaseMetrikaTask
from sandbox.projects.metrika.utils.codegen import edit_files
from sandbox.projects.metrika.utils.parameters import ArcadiaURL


class ClassAnalyzer(ast.NodeVisitor):
    def __init__(self):
        self.cls_childs = defaultdict(list)
        self.task_paths = {}
        self.task_names = {}
        self.path = None
        self.abstract = False
        self.abstracts = set()

    def visit_ClassDef(self, node):
        task_type = node.name
        if self.abstract:
            self.abstracts.add(task_type)

        for child in node.body:
            """
            class ClickHouseB2BTestRun:
                name = "CLICKHOUSE_B2B_TEST_RUN"  # child: target = value
            """
            if (
                isinstance(child, ast.Assign) and
                len(child.targets) == 1 and child.targets[0].id == 'name' and
                isinstance(child.value, ast.Str)
            ):
                task_name = child.value.s
                break
        else:
            task_name = ident(node.name)
        self.task_names[task_type] = task_name

        prev_task_path = self.task_paths.get(task_type, '')
        if len(prev_task_path) < len(self.path):
            for base in node.bases:
                self.cls_childs[base.attr if isinstance(base, ast.Attribute) else base.id].append(task_type)
            self.task_paths[task_type] = self.path
            logging.debug('Added %s from %s', task_type, self.path)
        else:
            logging.debug('Skipping %s from %s', task_type, self.path)

    def find_all_binary_tasks(self):
        # bfs
        logging.debug(json.dumps(self.cls_childs, indent=2))
        stack = [LastBinaryTaskRelease.__name__, LastRefreshableBinary.__name__]
        childs = []
        while stack:
            cls = stack.pop()
            cls_childs = self.cls_childs.get(cls)
            if cls_childs:
                childs.extend(cls_childs)
                stack.extend(cls_childs)

        return {self.task_names[task_type]: self.task_paths[task_type] for task_type in childs if task_type not in self.abstracts}


@with_parents
class MetrikaTasksList(BaseMetrikaTask):
    class Parameters(CommonParameters):
        arcadia_url = ArcadiaURL()
        tasks_path = sdk2.parameters.String('Директория тасок относительно рута', required=True, default='sandbox/projects/metrika')
        _binary = binary_task.binary_release_parameters_list(stable=True)

    class Context(sdk2.Task.Context):
        content = None
        branch = None
        pull_request_id = None

    @staticmethod
    def get_all_tasks(arcadia_url='arcadia-arc:/#trunk', tasks_path='sandbox/projects/metrika'):
        with vcs.mount_arc(arcadia_url) as arcadia_root:
            analyzer = ClassAnalyzer()
            for path in Path(arcadia_root).joinpath(tasks_path).glob('**/*.py'):
                logging.debug('Checking %s', path)
                program_name = path.parent.name
                analyzer.abstract = False
                if path.parent.joinpath('ya.make').exists():
                    m = re.search(r'(SANDBOX_PY23_TASK|PY23_LIBRARY)\((.*?)\)', path.parent.joinpath('ya.make').read_text(encoding='utf8'))
                    if m:
                        analyzer.abstract = (m.group(1) != 'SANDBOX_PY23_TASK')
                        if m.group(2):
                            program_name = m.group(2)
                analyzer.path = os.path.relpath(path.parent.joinpath(program_name).as_posix(), arcadia_root)
                analyzer.abstract = analyzer.abstract or (path.name != '__init__.py')
                if analyzer.abstract:
                    logging.debug('%s is abstract', program_name)
                tree = ast.parse(re.sub(r'^#.*coding(=|: )utf-?8.*', '', path.read_text(encoding='utf8'), count=1, flags=re.I))
                analyzer.visit(tree)

        return analyzer.find_all_binary_tasks()

    @property
    def arcanum_client(self):
        return arcanum_api.ArcanumApi(token=sdk2.Vault.data(settings.owner, settings.arcanum_token))

    def on_execute(self):
        with self.memoize_stage.content(commit_on_entrance=False):
            tasks = self.get_all_tasks(self.Parameters.arcadia_url, self.Parameters.tasks_path)
            self.Context.content = {'tasks': {task: os.path.dirname(path) for task, path in tasks.items()}}

        with self.memoize_stage.branch(commit_on_entrance=False):
            changes = {
                'sandbox/projects/metrika/utils/tasks/__init__.py': render("tasks.py.jinja2", self.Context.content),
                'sandbox/projects/metrika/a.yaml': render("a.yaml.jinja2", self.Context.content)
            }
            self.Context.branch = edit_files(self, "arcadia-arc:/#trunk", changes, "Update binary Sandbox-tasks list")

        if self.Context.branch:
            with self.memoize_stage.pr(commit_on_entrance=False):
                payload = {
                    "vcs": {
                        "from_branch": self.Context.branch,
                        "to_branch": "trunk"
                    },
                    "summary": "Update binary tasks list",
                    "auto_merge": "on_satisfied_requirements",
                    "on_new_diffset": {
                        "publish": True
                    }
                }
                response = self.arcanum_client.pr_create(data=payload, fields=["id", "url"])
                self.Context.pull_request_id = response["data"]["id"]
                self.set_info('<a href="{url}">Ревью {id}</a>'.format(url=response["data"]["url"], id=response["data"]["id"]), do_escape=False)
                self.arcanum_client.update_review_checks(
                    rr_id=self.Context.pull_request_id, content={"system": "arcanum", "type": "approved", "required": "False"}
                )

            with self.memoize_stage.pr_merged(commit_on_entrance=False, commit_on_wait=False):
                response = self.arcanum_client.pr_get(id=self.Context.pull_request_id, fields=["status"])
                status = response["data"]["status"]
                if status == "merged":
                    self.set_info("Merged")
                elif status in ["conflicts", "errors", "merge_failed", "discarded", "fatal_error"]:
                    raise errors.TaskFailure(status)
                else:
                    raise sdk2.WaitTime(180)
