# coding=utf-8
import datetime
import logging
import os

import requests

import json
from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common.juggler import jclient
from sandbox.projects.metrika.utils import CommonParameters
from sandbox.projects.metrika.utils import parameters
from sandbox.projects.metrika.utils.base_metrika_task import with_parents, BaseMetrikaTask
from sandbox.projects.metrika.utils.mixins.juggler_reporter import JugglerReporterMixin

BLACKBOX_URI = "http://blackbox-mimino.yandex.net/blackbox?method=oauth&userip=127.0.0.1&format=json"


@with_parents
class TokensChecker(BaseMetrikaTask, JugglerReporterMixin):
    """
    Осуществляет проверку валидности OAuth-токенов в секретнице. Репортит в Juggler, если срок валидности подходит к концу.
    """

    class Parameters(CommonParameters):
        container = parameters.LastPeasantContainerResource("Environment container resource", required=True)
        secrets = sdk2.parameters.List("Список секретов для проверки", required=True,
                                       description="Каждый элемент списка должен быть идентификатором секрета в секретнице. Проверяется последняя версия данного секрета. "
                                                   "Считается, что по ключам лежат значения в формате json, в которых может быть ключ 'oauth_token', чьё значение проверяется на валидность.")
        threshold_warn = sdk2.parameters.Integer("Порог предупреждения", required=True, default=10,
                                                 description="Если количество дней до истечения срока годности токена менее указанного здесь значения, то будет отправлен WARN.")
        threshold_crit = sdk2.parameters.Integer("Порог критичности", required=True, default=2,
                                                 description="Если количество дней до истечения срока годности токена менее указанного здесь значения, то будет отправлен CRIT.")
        yav_token = sdk2.parameters.Vault("OAuth-токен для доступа к секретнице", required=True, default="METRIKA:robot-metrika-test-yav")

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def on_execute(self):
        # "status":
        #   {
        #     "value":"VALID",
        #     "id":0
        #   },
        # 0 (VALID) — токен активен.
        # 4 (DISABLED) — OAuth-токен активен, но аккаунт, к которому он дает доступ, заблокирован.
        # 5 (INVALID) — OAuth-токен сформирован неверно, или просрочен.
        # если токен валиден - посмотреть на expire_time в блоке oauth. (Если поля нет - значит, токен сам по себе от старости не умрёт)
        # expire_time - 2019-06-27 04:40:22
        # Время, когда срок жизни токена закончится, если он не будет продлен.
        os.environ.update({"YAV_TOKEN": self.Parameters.yav_token.data()})
        for secret_uuid in self.Parameters.secrets:
            self._check_secret(secret_uuid)

    def _check_secret(self, secret_uuid):
        secret = json.loads(sdk2.helpers.subprocess.check_output(['yav', 'get', 'version', secret_uuid, '--json']))
        name = secret["secret_name"]
        msg = "Проверяем секрет {} ({})".format(name, secret_uuid)
        logging.info(msg)
        jstatus = "OK"
        jdescription = []
        for k, content in secret["value"].iteritems():
            logging.info("Проверяем ключ: {}".format(k))
            data = json.loads(content)
            if 'oauth_token' in data:
                token = data['oauth_token']
                resp = requests.get(BLACKBOX_URI, headers={'Authorization': 'OAuth {}'.format(token)}).json()
                logging.debug(resp)
                if 'status' in resp:
                    logging.info(resp['status'])
                    if resp['status']['id'] == 0:
                        # проверяем сколько осталось дней
                        if 'expire_time' in resp['oauth']:
                            logging.info(resp['oauth']['expire_time'])
                            left = datetime.datetime.strptime(resp['oauth']['expire_time'], "%Y-%m-%d %H:%M:%S") - datetime.datetime.now()
                            logging.info("{} days left".format(left.days))
                            if left.days <= self.Parameters.threshold_warn:
                                jstatus = "WARN" if jstatus == "OK" else jstatus
                                jdescription.append("{} - {} days left".format(k, left.days))
                            elif left.days <= self.Parameters.threshold_crit:
                                jstatus = "CRIT"
                                jdescription.append("{} - {} days left".format(k, left.days))
                        else:
                            logging.info("токен сам по себе от старости не умрёт")
                    else:
                        # Токен не годен или аккаунт заблокирован
                        jstatus = "CRIT"
                        jdescription.append("{} - {}".format(k, resp['error']))
                else:
                    logging.info(resp['error'])
                    jstatus = "CRIT"
                    jdescription.append(resp['error'])
            else:
                logging.info("No oauth_token found")

        logging.info("{} {}".format(jstatus, jdescription))
        self.set_info('<a href="https://yav.yandex-team.ru/secret/{id}">{id}</a>'.format(id=secret_uuid) + "\n" + "\n".join(jdescription), do_escape=False)
        jclient.send_events_to_juggler(secret_uuid, self.type.name, jstatus, 'https://yav.yandex-team.ru/secret/{id}'.format(id=secret_uuid) + "\n" + "\n".join(jdescription))
