# coding=utf-8
import collections
import json
import logging
from datetime import datetime
from distutils import version as distutils_version
from os import path

from sandbox import common, sdk2
from sandbox.common.types import task as common_task
from sandbox.projects.common import binary_task
from sandbox.projects.common.decorators import retries
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.admins import programs_list
from sandbox.projects.metrika.utils import arcanum_api, base_metrika_task, resource_types, settings
from sandbox.projects.metrika.utils.mixins import juggler_reporter
from sandbox.projects.metrika.utils.task_types import ya_package
from sandbox.sdk2 import parameters

PROJECTS = {
    "metrika-core": {"path": "metrika/core/programs", "package_type": "tarball", "ignore": ["test-daemon"], "juggler": "core"},
    "metrika-java": {"path": "metrika/java/api", "package_type": "docker", "ignore": [], "juggler": "java"}
}

DATABASES = {
    "versions_prod": {"juggler": "production"},
    "versions_test": {"juggler": "testing"}
}

ENVIRONMENTS = [
    "testing",
    "prestable",
    "production"
]


@base_metrika_task.with_parents
class MetrikaInfraVersionsRefresh(base_metrika_task.BaseMetrikaTask, juggler_reporter.JugglerReporterMixin):
    """
    Обновление списка версий демонов Метрики
    """

    class Context(sdk2.Context):
        daemons = {}

    class Parameters(utils.CommonParameters):
        description = "Обновление списка версий демонов Метрики"

        deploy_project = parameters.String("Проект в Деплое", required=True, default=PROJECTS.keys()[0], choices=[(item, item) for item in PROJECTS.keys()],
                                           description="Проект, версии которого получаюся")

        versions_database = parameters.String("База в PostgreSQL", required=True, default=DATABASES.keys()[0], choices=[(item, item) for item in DATABASES.keys()],
                                              description="База данных, в которую будут записаны версии")

        with parameters.Group("Параметры подключения к Postgresql") as postgresql_group:
            postgresql_cluster_hosts = parameters.List(
                "Список хостов кластера Postgresql",
                default=["vla-2xbvzkfutgmvxo3i.db.yandex.net", "man-szpxal2d5jw0n1ls.db.yandex.net", "sas-17qy960sguan4y80.db.yandex.net"],
                required=True
            )

            postgresql_user = parameters.String("Имя пользователя для подключения", required=True, default="metrika_infrastructure")

            postgresql_port = parameters.Integer("Номер порта для поключения", required=True, default=6432)

        with parameters.Group("Секреты") as secrets_group:
            tokens_secret = parameters.YavSecret("Секрет с токенами", required=True, default=settings.yav_uuid)

            deploy_token_key = parameters.String("Ключ токена Деплоя в секрете", required=True, default="deploy-token")

            versions_secret = parameters.YavSecret("Секрет для Versions", required=True, default="sec-01fgkrvrwrpw6q9322ntnvn1qr")

            postgresql_password_key = parameters.String("Ключ пароля PostgreSQL в секрете", required=True, default="pg_password")

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def on_execute(self):
        self.juggler_host = "metrika-sandbox-{}-{}".format(PROJECTS.get(self.Parameters.deploy_project).get("juggler"), DATABASES.get(self.Parameters.versions_database).get("juggler"))

        self.Context.daemons = collections.defaultdict(dict, self.Context.daemons)

        self.get_daemons_versions()

        self.save_daemons_versions()

    @property
    def deploy_client(self):
        import metrika.pylib.deploy.client as deploy
        return deploy.DeployAPI(token=self.Parameters.tokens_secret.data().get(self.Parameters.deploy_token_key))

    @property
    def arcanum_client(self):
        return arcanum_api.ArcanumApi(token=sdk2.Vault.data(settings.owner, settings.arcanum_token))

    def get_daemons_versions(self):
        from metrika.pylib import utils as pylib_utils

        project_stages = self.deploy_client.stage.list_project_stages(self.Parameters.deploy_project)
        exceptions = pylib_utils.parallel(self.get_deploy_info, queue=project_stages, thread_count=20, join_timeout=600)
        if exceptions:
            logging.exception("Got exceptions from Deploy")
            raise common.errors.TaskFailure("\n".join(str(exception) for exception in exceptions))

        all_programs = programs_list.MetrikaProgramsList.get_all_programs("arcadia-arc:/#trunk", PROJECTS.get(self.Parameters.deploy_project).get("path"))
        project_programs = [daemon for daemon in all_programs.keys() if programs_list.PACKAGE_JSON in all_programs.get(daemon)]
        exceptions = pylib_utils.parallel(self.get_conductor_info, queue=project_programs, thread_count=20, join_timeout=600)
        if exceptions:
            logging.exception("Got exceptions from Conductor")
            raise common.errors.TaskFailure("\n".join(str(exception) for exception in exceptions))

        for daemon in self.Context.daemons.values():
            daemon["is_changed"] = self.is_changed(daemon)
            daemon["author"] = daemon.get("testing", {}).get("author") or daemon.get("production", {}).get("author")

        self.Context.daemons = list(sorted(self.Context.daemons.values(), key=self.get_sort_key, reverse=True))

    def save_daemons_versions(self):
        import psycopg2

        connection = psycopg2.connect(
            target_session_attrs="read-write",
            host=",".join(self.Parameters.postgresql_cluster_hosts),
            port=self.Parameters.postgresql_port,
            user=self.Parameters.postgresql_user,
            password=self.Parameters.versions_secret.data().get(self.Parameters.postgresql_password_key),
            database=self.Parameters.versions_database
        )

        try:
            with connection.cursor() as cursor:
                sql = "update versions_daemons set updated_at = %s, daemons = %s where project = %s"
                cursor.execute(sql, (datetime.now(), json.dumps(self.Context.daemons), self.Parameters.deploy_project))
                connection.commit()
        except Exception as err:
            logging.exception("Got exception")
            raise common.errors.TaskFailure(str(err))
        else:
            logging.debug("Sql is finished")
        finally:
            connection.close()

    def get_deploy_info(self, stage):
        name, environment = stage.get("id").rsplit("-", 1)
        if environment in ENVIRONMENTS:
            version = self.get_mtapi_version(name, environment)
            build_task = self.get_build_task(PROJECTS.get(self.Parameters.deploy_project).get("package_type"), version)
            author = self.get_author_from_build_task(build_task)
            self.Context.daemons[name]["name"] = name
            self.Context.daemons[name][environment] = {"version": version, "author": author, "link": "https://deploy.yandex-team.ru/stages/{}".format(stage.get("id"))}
            self.Context.daemons[name]["is_deploy"] = True
            if environment == "production":
                self.fill_last_merged_from_build_task(build_task, name)

    def get_conductor_info(self, program):
        package = "{}-metrika-yandex".format(program)

        for environment in ENVIRONMENTS:
            version = self.get_mtapi_version(package, environment)
            if version:
                build_task = self.get_build_task("debian", version)
                author = self.get_author_from_build_task(build_task)
                self.Context.daemons[program]["name"] = program
                self.Context.daemons[program][environment] = {"version": version, "author": author, "link": "https://c.yandex-team.ru/packages/{}/tickets".format(package)}
                self.Context.daemons[program]["is_deploy"] = False
                if environment == "production":
                    self.fill_last_merged_from_build_task(build_task, program)

    def fill_last_merged_from_build_task(self, build_task, name):
        subpath = "/{}/".format(name)
        if build_task and subpath in build_task.Parameters.packages:
            packages = build_task.Parameters.packages.split(";")
            package_path = next(package for package in packages if subpath in package)
            if package_path:
                daemon_path = path.dirname(package_path)
                self.Context.daemons[name]["path"] = daemon_path
                try:
                    package_arcanum_tree = self.arcanum_client.get_tree(daemon_path)
                    self.Context.daemons[name]["last_merged"] = package_arcanum_tree.get("lastChangedDate")
                except Exception as err:
                    logging.exception("Unhandled exception during lastChangedDate retrieving: {}".format(err))

    def get_sort_key(self, daemon):
        try:
            return self.is_changed(daemon), self.get_all_versions(daemon), daemon["name"]
        except Exception as err:
            logging.exception("Unhandled exception during sorting: {}".format(err))
            return (True,)

    def is_changed(self, daemon):
        return len(self.get_all_versions(daemon)) != 1

    def get_all_versions(self, daemon):
        versions = [self.get_version(daemon, "testing"), self.get_version(daemon, "prestable"), self.get_version(daemon, "production")]
        return {tuple(distutils_version.LooseVersion(version).version) for version in versions if version and version[0].isdigit()}

    @staticmethod
    @retries(3)
    def get_mtapi_version(name, env):
        from metrika.pylib.mtapi.packages import PackagesAPI, BadRequestException

        mtapi_packages = PackagesAPI()
        mtapi_packages._verify = False

        try:
            return mtapi_packages.pkg_version_per_environment(env=env, pkg_name=name)[0].pkg_version
        except BadRequestException:
            logging.exception("Version for '{}' was not found.".format(name))
            return
        except Exception as err:
            logging.exception("Unhandled exception during mtapi_packages.pkg_version_per_environment(env='{}', pkg_name='{}') call: {}".format(env, name, err))

    @staticmethod
    def get_build_task(package_type, version):
        return sdk2.Task.find(
            task_type=ya_package.MetrikaYaPackage,
            status=common_task.Status.Group.SUCCEED,
            children=True,
            input_parameters={
                "resource_type": resource_types.BaseMetrikaBinaryResource.name,
                "package_type": package_type,
                "custom_version": version
            }
        ).first()

    @staticmethod
    def get_author_from_build_task(build_task):
        if build_task:
            return build_task.author

    @staticmethod
    def get_version(daemon, environment):
        return daemon.get(environment, {}).get("version") or ""
