# coding=utf-8

from sandbox.projects.common import binary_task
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.core.metrika_core_development import development_helper, development_info
from sandbox.projects.metrika.utils import base_metrika_task
from sandbox.projects.metrika.utils.mixins import juggler_reporter
from sandbox.projects.metrika.utils.parameters import hide
from sandbox.projects.metrika.utils.pipeline import pipeline
from sandbox.sdk2 import parameters

REPORT_TTL = 30


@base_metrika_task.with_parents
class MetrikaCoreDevelopment(pipeline.PipelineBaseTask, juggler_reporter.JugglerReporterMixin):
    """
    Сборка и тестирование демонов Движка Метрики
    """

    @property
    def development(self):
        return development_info.DevelopmentInfo(self.Context.development_info_state)

    class Context(pipeline.PipelineBaseTask.Context):
        development_info_state = development_info.DevelopmentInfo().state
        run_test_tasks_ids = []
        build_packages_task_ids = []

    class Parameters(utils.CommonParameters):
        description = "Сборка и тестирование демонов Движка Метрики"

        tracker_issue = parameters.String(
            "Ключ тикета", required=False, default="",
            description="Ключ тикета в Трекере."
        )

        arcadia_url = parameters.ArcadiaUrl(
            "URL Аркадии", required=True, default="",
            description="Ветка или коммит, из которой будут собраны демона и запущены тесты. Переменная '{tracker_issue}' подставляется автоматически."
        )

        with parameters.Group("Настройки этапа сборки") as build_packages_group:
            build_packages = parameters.Bool(
                "Сборка", required=True, default=True,
                description="Собирать ли новые, или указать уже собранные версии пакетов."
            )

            with build_packages.value[True]:
                daemon_list = parameters.List(
                    "Список демонов", required=True,
                    description="Демона Движка Метрики, сборка и тестирование которых будет осуществлено."
                )

            with build_packages.value[False]:
                daemon_dictionary = parameters.Dict(
                    "Список демонов", required=True, default={},
                    description="Демона Движка Метрики, тестирование которых будет осуществлено."
                )

        with parameters.Group("Настройки этапа тестирования") as run_tests_group:
            run_tests = parameters.Bool(
                "Тестирование", required=True, default=True,
                description="Запускать ли тесты."
            )

            with run_tests.value[True]:
                run_b2b_tests = parameters.Bool(
                    "B2B тестирование", required=True, default=True,
                    description="Запускать ли B2B тесты."
                )

                with run_b2b_tests.value[True]:
                    fail_b2b_task_on_test_failure = parameters.Bool('Фэйлить таску при падении B2B тестов', required=True, default=False)

                run_functional_tests = parameters.Bool(
                    "Функциональное тестирование", required=True, default=True,
                    description="Запускать ли функциональные тесты."
                )

        with parameters.Group("Настройки этапа отправки уведомлений") as send_notifications_group:
            send_notifications = parameters.Bool(
                "Уведомления", required=True, default=True,
                description="Отправлять ли уведомления."
            )

            with send_notifications.value[True]:
                send_telegram_notifications = parameters.Bool(
                    "Уведомления в Telegram", required=True, default=True,
                    description="Отправлять ли Telegram уведомления."
                )

                with send_telegram_notifications.value[True]:
                    telegram_notifications_recipients = parameters.List(
                        "Получатели Telegram уведомлений", required=False, default=[],
                        description="По умолчанию уведомление отправляется автору таски или в общий канал."
                    )

                    send_telegram_notifications_on_success = parameters.Bool(
                        "Уведомления об успехе", required=False, default=False,
                        description="Отправлять ли Telegram уведомления об успешнно пройденных тестах."
                    )

                send_email_notifications = parameters.Bool(
                    "Уведомления по e-mail", required=True, default=False,
                    description="Отправлять ли e-mail уведомления."
                )

                with send_email_notifications.value[True]:
                    email_notifications_recipients = parameters.List(
                        "Получатели e-mail уведомлений", required=False, default=[],
                        description="Могут быть указаны как люди, так и рассылки. "
                                    "По умолчанию уведомление отправляется автору."
                    )

                    send_email_notifications_on_success = parameters.Bool(
                        "Уведомления об успехе", required=False, default=False,
                        description="Отправлять ли e-mail уведомления об успешнно пройденных тестах."
                    )

        _binary = hide(binary_task.binary_release_parameters_list(stable=True))

    def on_save(self):
        self.development.author = self.real_author

        if not self.Parameters.arcadia_url:
            self.Parameters.arcadia_url = "arcadia-arc:/#users/{}/{{tracker_issue}}".format(self.development.author)

        if self.Parameters.tracker_issue:
            self.Parameters.arcadia_url = self.Parameters.arcadia_url.format(tracker_issue=self.Parameters.tracker_issue)

        development_helper.DevelopmentHelper.check_daemons(self, self.Parameters.daemon_list if self.Parameters.build_packages else self.Parameters.daemon_dictionary)

    def on_enqueue(self):
        self.development.arcadia_url = self.Parameters.arcadia_url

        daemon_list = self.Parameters.daemon_list if self.Parameters.build_packages else self.Parameters.daemon_dictionary.keys()

        development_helper.DevelopmentHelper.update_description(self, self.development.author, daemon_list, self.development.arcadia_url)

        development_helper.DevelopmentHelper.set_build_description(self)

        development_helper.DevelopmentHelper.update_tests_description(self, self.Context.run_test_tasks_ids)

    def create_stages(self):
        return [
            (self.stage_build_packages, "Сборка"),
            (self.stage_run_tests, "Тесты"),
            (self.stage_send_notifications, "Уведомления")
        ]

    def stage_build_packages(self):
        if self.Parameters.build_packages:
            version = "1.{}".format(self.id)
            self.development.daemons = dict((daemon, version) for daemon in self.Parameters.daemon_list)

            build_tasks = development_helper.DevelopmentHelper.get_build_tasks(
                self, self.development.author, self.Parameters.daemon_list, version, self.development.arcadia_url
            )

            self.run_subtasks(build_tasks, subtasks_variable=self.Context.build_packages_task_ids, info=False)

            development_helper.DevelopmentHelper.set_build_info(self, self.Context.build_packages_task_ids, repo="metrika-trusty")
        else:
            self.development.daemons = dict((daemon, version or development_helper.DevelopmentHelper.get_stable_version(daemon)) for daemon, version in self.Parameters.daemon_dictionary.iteritems())

            development_helper.DevelopmentHelper.set_build_packages_info(self, self.development.daemons, repo="metrika-trusty")

        development_helper.DevelopmentHelper.update_build_description(self, self.Context.build_packages_task_ids)
        development_helper.DevelopmentHelper.set_build_description(self)

    def stage_run_tests(self):
        self.development.tests = development_helper.DevelopmentHelper.map_tests(self.development.daemons)
        if self.Parameters.run_tests and (self.Parameters.run_b2b_tests or self.Parameters.run_functional_tests):
            test_tasks = development_helper.DevelopmentHelper.get_all_tests_tasks(
                self.development, self.development.arcadia_url, REPORT_TTL,
                self.Parameters.tracker_issue, self.Parameters.run_b2b_tests, self.Parameters.run_functional_tests,
                fail_b2b_task_on_test_failure=self.Parameters.fail_b2b_task_on_test_failure
            )
            self.run_subtasks(test_tasks, subtasks_variable=self.Context.run_test_tasks_ids, after_subtasks_enqueued=self.after_tests_enqueued, info=False)
        else:
            self.set_info("Запуска тестов не будет.")

    def after_tests_enqueued(self):
        development_helper.DevelopmentHelper.set_run_tests_info(self, self.Context.run_test_tasks_ids)
        development_helper.DevelopmentHelper.update_tests_description(self, self.Context.run_test_tasks_ids)

    def stage_send_notifications(self):
        all_tests_passed = development_helper.DevelopmentHelper.all_tests_passed(self.Context.run_test_tasks_ids)

        send_telegram_notifications = self.Parameters.send_telegram_notifications and (self.Parameters.send_telegram_notifications_on_success or not all_tests_passed)

        send_email_notifications = self.Parameters.send_email_notifications and (self.Parameters.send_email_notifications_on_success or not all_tests_passed)

        send_notifications = self.Parameters.send_notifications and (send_telegram_notifications or send_email_notifications)

        if send_notifications:
            if send_telegram_notifications:
                development_helper.DevelopmentHelper.send_telegram_notifications(self, self.Parameters.telegram_notifications_recipients, self.Context.run_test_tasks_ids)

            if send_email_notifications:
                development_helper.DevelopmentHelper.send_email_notifications(self, self.Parameters.email_notifications_recipients, self.Context.run_test_tasks_ids)

            development_helper.DevelopmentHelper.set_send_notifications_info(self, send_telegram_notifications, send_email_notifications)
        else:
            self.set_info("Отправки уведомлений не будет.")
