# coding=utf-8
from __future__ import unicode_literals

import logging
import os

from sandbox import sdk2
from sandbox.sdk2 import parameters
from sandbox.projects.common import binary_task
from sandbox.projects.metrika import utils as metrika_utils
from sandbox.projects.metrika.admins.clickhouse import clickhouse_statbox_traf_tests_pipeline
from sandbox.projects.metrika.core import metrika_core_arcadia_tests_run, metrika_core_b2b_tests_run, metrika_core_dicts_upload
from sandbox.projects.metrika.core.metrika_core_dicts_upload import MetrikaDictsUpload
from sandbox.projects.metrika.core.metrika_core_dicts_upload import lib as upload_lib
from sandbox.projects.metrika.core.utils import metrika_core_helper
from sandbox.projects.metrika.java.utils import metrika_java_helper
from sandbox.projects.metrika.utils import base_metrika_task, parameters as metrika_parameters
from sandbox.projects.metrika.utils import settings
from sandbox.projects.metrika.utils.parameters import DataCenterParameters
from sandbox.projects.metrika.utils.pipeline import pipeline
from sandbox.projects.release_machine.helpers import arcanum_helper

REPORT_TTL = 730


@base_metrika_task.with_parents
class MetrikaDictsRelease(pipeline.PipelineBaseTask):
    TASK_TEMPLATE = """\
((https://wiki.yandex-team.ru/jandexmetrika/operations/Vykladka-slovarejj-mtacs/ Релизный процесс словарей mtacs))

Задача в SB: https://sandbox.yandex-team.ru/task/{sandbox_task}
Релизный тикет: https://st.yandex-team.ru/{ticket}
Ревью: https://a.yandex-team.ru/arc/review/{review_id}/details

Затронутые словари: https://sandbox.yandex-team.ru/resource/{dicts_resource_id}
{dicts}

Раскатите, пожалуйста, их на ((https://sandbox.yandex-team.ru/task/{testing} тест)) и ((https://sandbox.yandex-team.ru/task/{stable} прод)).
"""

    class Parameters(metrika_utils.CommonParameters):
        description = 'Релиз словарей Метрики'

        name = parameters.String('Название словарей', required=True, description='Январь 2007')
        ticket = metrika_parameters.TrackerIssue(label='Задача', required=True)
        review_id = parameters.Integer('ID ревью', description='42069, если не указано, то trunk')

        dicts_from_review = parameters.Bool('Все словари из ревью', default=True, required=True, description='Словарями считаются все файлы из ревью')
        with dicts_from_review.value[False]:
            dicts = parameters.List(
                'Пути до словарей', required=True,
                description='Список измененных словарей из ревью (например metrika/core/db_dumps/Metrica/Messengers.sql)'
            )

        data_center_params = DataCenterParameters()

        _binary = metrika_parameters.hide(binary_task.binary_release_parameters_list(stable=True))

    def create_stages(self):
        return [
            (self.stage_release_issue, 'Релизный тикет'),
            (self.prepare_dicts, 'Словари ClickHouse'),
            (self.stage_run_tests, 'Тестирование'),
            (self.stage_dicts_issue, 'Создание тикета на выкладку')
        ]

    def stage_release_issue(self):
        search_parameters = {
            'queue': 'METR',
            'type': 'release',
            'tags': 'metrika-dicts-release-{}'.format(self.Parameters.review_id or self.Parameters.name.replace(' ', '_'))
        }
        create_parameters = search_parameters.copy()
        create_parameters.update({
            'summary': 'Релиз словарей. {}'.format(self.Parameters.name),
            'assignee': self.author,
            'links': [{
                'relationship': 'relates',
                'issue': self.Parameters.ticket
            }]
        })

        self.Context.release_issue_key = metrika_java_helper.MetrikaJavaHelper.find_or_create_issue(self.st_client, search_parameters, create_parameters).key

        logging.info('Issue {} created.'.format(self.Context.release_issue_key))
        self.set_info('Релизный тикет: <a href="https://st.yandex-team.ru/{0}">{0}</a>'.format(self.Context.release_issue_key), do_escape=False)

    def prepare_dicts(self):
        dicts_dir = self.wd('dicts')
        os.makedirs(dicts_dir)
        upload_lib.prepare_dicts(dicts_dir, review_id=self.Parameters.review_id, changed_dicts=None if self.Parameters.dicts_from_review else self.Parameters.dicts)

        dicts_res = metrika_core_dicts_upload.DictsReleaseResource(self, self.Parameters.name, dicts_dir)
        sdk2.ResourceData(dicts_res).ready()
        self.Context.dicts_resource_id = dicts_res.id

        self.Context.dicts = os.listdir(dicts_dir)
        self.set_info(
            '<a href="https://sandbox.yandex-team.ru/resource/{}">Cловари</a>:\n{}'.format(
                self.Context.dicts_resource_id,
                '\n'.join(self.Context.dicts)
            ),
            do_escape=False
        )

        # TODO здесь где-то нужно подготовить ресурс с TSV-вариантом этих словарей для использования в тестах трафа

    def stage_run_tests(self):
        arcanum_client = arcanum_helper.ArcanumApi(token=sdk2.Vault.data(settings.owner, settings.arcanum_token))
        branch = (
            arcanum_client.get_review_request(self.Parameters.review_id)['branch'] or 'trunk'
            if self.Parameters.review_id else
            'trunk'
        )

        common_test_params = dict(
            report_startrek=True,
            issue_key=self.Context.release_issue_key,
            arcadia_url=metrika_java_helper.MetrikaJavaHelper.get_arcadia_url(branch),
            fail_task_on_test_failure=True,
        )
        tests_tasks = [
            (
                metrika_core_arcadia_tests_run.MetrikaCoreArcadiaTestsRun,
                dict(
                    common_test_params,
                    checkout_arcadia_from_url=metrika_java_helper.MetrikaJavaHelper.get_arcadia_url(branch),
                    targets="metrika/core/db_dumps/tests"
                )
            ),
            (
                metrika_core_b2b_tests_run.MetrikaCoreB2bTestsRun.name,
                dict(
                    common_test_params,
                    force_test_scenario=True,
                )
            ),
            (
                clickhouse_statbox_traf_tests_pipeline.ClickHouseStatboxTrafTestsPipeline.name,
                dict(
                    report_startrek=True,
                    issue_key=self.Context.release_issue_key,
                    caas_parent='days-32_sample-4',
                    ch_dicts_resource=self.Context.dicts_resource_id,
                    tolerance=True,
                    max_abs_diff=1e-4,
                    data_center=self.Parameters.data_center
                )
            )
        ]
        tests_tasks.extend(metrika_core_helper.MetrikaCoreHelper.get_functional_tests_tasks(
            ["logprocessd"],
            metrika_java_helper.MetrikaJavaHelper.get_arcadia_url(branch),
            REPORT_TTL,
            self.Context.release_issue_key
        ))
        self.run_subtasks(tests_tasks)

    def stage_dicts_issue(self):
        from metrika.pylib import duty
        assignee = duty.DutyAPI(token=sdk2.Vault.data(settings.owner, settings.infra_token)).get_duty_group('metrika', 'core')['duty']

        logging.debug(assignee)

        self.Context.release_tasks = {}
        for env, _ in metrika_core_dicts_upload.MetrikaDictsUpload.Parameters.env_type.choices:
            self.Context.release_tasks[env] = metrika_core_dicts_upload.MetrikaDictsUpload(
                self,
                dicts_source='resource',
                dicts_resource=self.Context.dicts_resource_id,
                env_type=env,
                is_mdb=True,
                ticket=self.Context.release_issue_key,
                description='{} [{}]'.format(self.Parameters.name, env)
            ).id

        create_parameters = {
            'queue': 'METR',
            'summary': 'Выложить справочники. {}'.format(self.Parameters.name),
            'type': 'task',
            'assignee': assignee,
            'followers': [self.author],
            'parent': self.Context.release_issue_key,
            'description': self.TASK_TEMPLATE.format(
                ticket=self.Context.release_issue_key,
                review_id=self.Parameters.review_id,
                sandbox_task=self.id,
                dicts='\n'.join(sorted(self.Context.dicts)),
                dicts_resource_id=self.Context.dicts_resource_id,
                **self.Context.release_tasks
            )
        }

        self.Context.issue_key = self.st_client.issues.create(**create_parameters).key
        logging.info('Issue {} created.'.format(self.Context.issue_key))

        for task_id in self.Context.release_tasks.values():
            task = sdk2.Task[task_id]  # type: MetrikaDictsUpload
            task.Parameters.deploy_ticket = self.Context.issue_key
            task.save()

        self.set_info('Тикет на выкладку: <a href="https://st.yandex-team.ru/{0}">{0}</a>'.format(self.Context.issue_key), do_escape=False)
