# coding=utf-8
from __future__ import unicode_literals

import datetime
import logging
import os
import time

from sandbox import sdk2
from sandbox.common import patterns
from sandbox.common.types import resource as ctr
from sandbox.projects.common import binary_task
from sandbox.projects.metrika import utils as metrika_utils
from sandbox.projects.metrika.admins.infra import metrika_infra_upload
from sandbox.projects.metrika.admins.infra.api import infra_api
from sandbox.projects.metrika.core.metrika_core_dicts_upload import lib
from sandbox.projects.metrika.utils import base_metrika_task, parameters as metrika_parameters, settings
from sandbox.projects.metrika.utils import resource_types as metrika_resource_types
from sandbox.projects.metrika.utils.mixins import juggler_reporter
from sandbox.sdk2 import parameters


class MetrikaDictsUploadRestartException(Exception):
    pass


class DictsReleaseResource(sdk2.Resource):
    name = 'DICTS_RELEASE_RESOURCE'
    releasable = True
    releasers = metrika_resource_types.METRIKA_RELEASERS
    restart_policy = ctr.RestartPolicy.IGNORE


@base_metrika_task.with_parents
class MetrikaDictsUpload(base_metrika_task.BaseMetrikaTask, juggler_reporter.JugglerReporterMixin):
    RESTARTED_TAG = 'DAEMONS_RESTARTED'
    METRIKA_INFRA_SERVICE_ID = 318
    METRIKA_INFRA_ENV_IDS = {
        'stable': 466,
        'testing': 467
    }

    class Parameters(metrika_utils.CommonParameters):
        with parameters.RadioGroup('Источник словарей') as dicts_source:
            dicts_source.values.resource = dicts_source.Value('Ресурс', default=True)
            dicts_source.values.review = dicts_source.Value('Ревью')
            dicts_source.values.revision = dicts_source.Value('Ревизия')

        with dicts_source.value['resource']:
            dicts_resource = parameters.Resource('Ресурс со словарями', resource_type=DictsReleaseResource)

        with dicts_source.value['review']:
            review_id = parameters.Integer('ID ревью', required=True, description='42069')
            dicts_from_review = parameters.Bool('Все словари из ревью', default=True, required=True)

            with dicts_from_review.value[False]:
                review_dicts = parameters.List(
                    'Пути до словарей', required=True,
                    description='Список измененных словарей из ревью (например metrika/core/db_dumps/Metrica/Messengers.sql)'
                )

        with dicts_source.value['revision']:
            arcadia_url = parameters.ArcadiaUrl('URL Аркадии', required=True, default='arcadia-arc:/#trunk')
            repo_dicts = parameters.List(
                'Пути до словарей', required=True,
                description='Список словарей (например metrika/core/db_dumps/Metrica/Messengers.sql)'
            )

        need_restart = parameters.Bool('Нужен рестарт демонов', default=True)

        with parameters.RadioGroup('env') as env_type:
            for env in {'testing', 'stable'}:
                env_type.values[env] = env_type.Value(env)

        ticket = metrika_parameters.TrackerIssue('Релизный тикет', required=False)

        deploy_ticket = metrika_parameters.TrackerIssue('Тикет на применение', required=False)

        _binary = metrika_parameters.hide(binary_task.binary_release_parameters_list(stable=True))

    def _juggler_predicate(self, status):
        return True

    def _get_dicts_dir(self):
        if self.Parameters.dicts_source == 'resource':
            return sdk2.ResourceData(self.Parameters.dicts_resource).path
        else:
            dicts_dir = self.path('wd', 'dicts')
            os.makedirs(dicts_dir.as_posix())
            lib.prepare_dicts(
                dicts_dir,
                arcadia_url=None if self.Parameters.dicts_source != 'revision' else self.Parameters.arcadia_url,
                review_id=None if self.Parameters.dicts_source != 'review' else self.Parameters.review_id,
                changed_dicts=(
                    (None if self.Parameters.dicts_from_review else self.Parameters.review_dicts)
                    if self.Parameters.dicts_source == 'review'
                    else self.Parameters.repo_dicts
                )
            )
            return dicts_dir

    @patterns.singleton_property
    def infra_api(self):
        return infra_api.InfraApi(token=sdk2.yav.Secret(settings.yav_uuid).data()['infra-token'])

    def on_save(self):
        self.Parameters.tags = []

    def on_execute(self):
        import pymysql

        dicts_dir = self._get_dicts_dir()
        port = 3306
        user = 'dba'
        password = sdk2.yav.Secret('sec-01ep6qbrjdxhpwrebchccqb8fz').data()['dba-password']
        database = 'conv_main'
        if self.Parameters.env_type == 'stable':
            host = 'mdb0juohhl77rb30l0ag.haproxy-prod.metrika.yandex.net'
        elif self.Parameters.env_type == 'testing':
            host = 'metrika-main.haproxy-test.metrika.yandex.net'

        with self.memoize_stage.infra_start(commit_on_entrance=False):
            self.Context.event_id = self.infra_api.events.create(metrika_infra_upload.Event(
                service_id=self.METRIKA_INFRA_SERVICE_ID,
                environment_id=self.METRIKA_INFRA_ENV_IDS[self.Parameters.env_type],
                title='Обновление словарей {}на {}'.format(
                    '"{}" '.format(self.Parameters.dicts_resource.description) if self.Parameters.dicts_source == 'resource' else '',
                    self.Parameters.env_type
                ),
                description='Задача в SB: https://sandbox.yandex-team.ru/task/{}'.format(self.id),
                start_time=int(time.mktime(datetime.datetime.now().timetuple())),
                tickets=self.Parameters.ticket,
                severity='minor',
                type='maintenance'
            ).to_dict()).get('id')
            self.set_info('<a href="https://infra.yandex-team.ru/event/{}">Событие в инфре</a>'.format(self.Context.event_id), do_escape=False)

        with self.memoize_stage.update_dicts(commit_on_entrance=False):
            connection = pymysql.connect(
                host=host, port=port,
                user=user, password=password,
                database=database,
                use_unicode=True
            )
            with connection.cursor() as cursor:
                cursor.execute('select @@hostname;')
                self.set_info('Обновление словарей на {}:\n'.format(cursor.fetchone()[0]) + '\n'.join(os.listdir(dicts_dir.as_posix())))

                for file in dicts_dir.glob('*'):
                    logging.debug('Updating table %s', file.stem)

                    for line in file.read_text(encoding='utf8').split(';\n'):
                        line = line.strip()
                        if line:
                            logging.debug(line)
                            cursor.execute(line)
                    connection.commit()

        if self.Parameters.need_restart and self.RESTARTED_TAG not in self.Parameters.tags:
            raise MetrikaDictsUploadRestartException(
                'Нужно рестартнуть демона, которые читают эти словари, и после добавить тег {tag} в таску\n'
                'Стейдж logprocessd: https://deploy.yandex-team.ru/stages/logprocessd-{environment}/edit/du-logprocessd/box-logprocessd/wl-logprocessd\n'
                'Проинкрементить RESTART_COUNTER'.format(
                    environment='production' if self.Parameters.env_type == 'stable' else 'testing',
                    tag=self.RESTARTED_TAG
                )
            )

        with self.memoize_stage.infra_end(commit_on_entrance=False):
            self.infra_api.events.update(self.Context.event_id, {'finishTime': int(time.mktime(datetime.datetime.now().timetuple()))})

        with self.memoize_stage.comment(commit_on_entrance=False):
            comment = 'Словари выложены на {}\nhttps://sandbox.yandex-team.ru/task/{}'.format(self.Parameters.env_type, self.id)
            if self.Parameters.ticket:
                self.st_client.issues[self.Parameters.ticket].comments.create(text=comment)
            if self.Parameters.deploy_ticket:
                self.st_client.issues[self.Parameters.deploy_ticket].comments.create(text=comment)
