# coding=utf-8
from __future__ import unicode_literals

import codecs
import json
import logging
from shutil import copyfile

import six
from pathlib2 import Path

from sandbox.projects.metrika.utils import vcs
from sandbox.sdk2.vcs import svn


def prepare_dicts(dicts_dst_dir, arcadia_url=None, review_id=None, changed_dicts=None):
    """
    Берет словари или из репозитория по `arcadia_url` или из ревью `review_id` или из транка и кладет в `dicts_dst_dir`

    :param str dicts_dst_dir:
    :param str arcadia_url
    :param int review_id:
    :param List[str] changed_dicts:
    :return:
    """

    if arcadia_url and not changed_dicts:
        raise ValueError('List of dicts is required when using Arcadia URL')

    dicts_dst_dir = Path(dicts_dst_dir)
    with vcs.mount_arc(arcadia_url or 'arcadia-arc:/#trunk') as arcadia_root:
        if review_id:
            dicts_dir = dicts_dst_dir.parent.joinpath('arcadia')
            dicts_dir.mkdir(parents=True, exist_ok=True)
            svn.Arcadia.apply_patch(arcadia_path=dicts_dir.as_posix(), arcadia_patch='arc:{}'.format(review_id), dest_dir=dicts_dir.as_posix())
        else:
            dicts_dir = Path(arcadia_root)

        if not changed_dicts:
            review_dicts = [f for f in dicts_dir.joinpath('metrika').glob('**/*') if f.is_file()]
        else:
            review_dicts = [dicts_dir.joinpath(dict) for dict in changed_dicts]
        dicts = []
        for dict in review_dicts:
            dicts.append(dict.name)
            if dict.suffix == '.json':
                logging.debug('Converting %s, json -> sql', dict.as_posix())
                ddl_file_path = 'metrika/core/db_dumps/Metrica/{}.sql'.format(dict.stem)
                for repo in [dicts_dir, arcadia_root]:
                    ddl_file = Path(repo).joinpath(ddl_file_path)
                    if ddl_file.exists():
                        break
                else:
                    raise Exception('{} not found'.format(ddl_file_path))

                _json_to_sql(dict, ddl_file, dicts_dst_dir)
            elif dict.suffix == '.sql':
                src, dst = dict.as_posix(), dicts_dst_dir.joinpath(dicts[-1]).as_posix()
                logging.debug('Copying %s -> %s', src, dst)
                copyfile(src, dst)
            else:
                raise ValueError('Unkown file extension: %s', dict.as_posix())


def _json_to_sql(file_path, ddl_file, dicts_dir):
    """
    :param Path file_path:
    :param Path ddl_file:
    :param Path dicts_dir:
    :return:
    """

    with codecs.open(file_path.as_posix(), encoding='utf8') as file:
        rows = file.readlines()

    table_name = file_path.stem
    sql_name = table_name + '.sql'
    with codecs.open(dicts_dir.joinpath(sql_name).as_posix(), mode='w', encoding='utf8') as sql:
        sql.write(ddl_file.read_text(encoding='utf8'))

        sql.write('\n')
        sql.write('LOCK TABLES `{}` WRITE;\n'.format(table_name))

        for row in rows:
            logging.debug(row)
            row = json_row_to_sql(row, table_name)
            if not row:
                continue

            sql.write(row)
            sql.write('\n')
        sql.write('UNLOCK TABLES;\n')


def json_row_to_sql(row, table_name):
    import pymysql

    row = (
        row.strip().strip(',')
            .replace('\\', r'\\')  # Invalid \escape: line 1 column 141 (char 140)
    )
    if not row:
        return

    parsed_row = json.loads(row)
    return 'INSERT INTO `{table_name}` ({columns}) VALUES ({values});'.format(
        table_name=table_name,
        columns=', '.join('`{}`'.format(col) for col in parsed_row.keys()),
        values=', '.join(
            "'{}'".format(pymysql.escape_string(six.text_type(value)))
            for value in parsed_row.values()
        )
    ).replace(r'\\', '\\')
