# coding=utf-8

from sandbox import sdk2
from sandbox.common.types import task as task_type
from sandbox.projects.common import binary_task
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.core.metrika_core_release import release_helper, release_info
from sandbox.projects.metrika.utils import base_metrika_task
from sandbox.projects.metrika.utils.mixins import releasable
from sandbox.projects.metrika.utils.parameters import hide, TrackerIssue
from sandbox.projects.metrika.utils.pipeline import pipeline
from sandbox.sdk2 import parameters

REPORT_TTL = 730


@base_metrika_task.with_parents
class MetrikaCoreRelease(pipeline.PipelineBaseTask, releasable.ReleasableMixin):
    """
    Релиз демонов Движка Метрики
    """

    @property
    def release(self):
        return release_info.ReleaseInfo(self.Context.release_info_state)

    @property
    def release_template(self):
        if not self.Context.released and self.status != task_type.Status.RELEASED:
            return sdk2.ReleaseTemplate(None, None, None, [task_type.ReleaseStatus.TESTING])
        else:
            return sdk2.ReleaseTemplate(None, None, None, [task_type.ReleaseStatus.STABLE, task_type.ReleaseStatus.PRESTABLE, task_type.ReleaseStatus.TESTING])

    class Context(pipeline.PipelineBaseTask.Context):
        release_info_state = release_info.ReleaseInfo().state
        run_test_tasks_ids = []
        build_packages_task_ids = []

    class Parameters(utils.CommonParameters):
        description = "Релиз демонов Движка Метрики"

        tracker_issue = TrackerIssue(required=True)

        arcadia_url = parameters.ArcadiaUrl(
            "URL Аркадии", required=True, default="",
            description="Ветка или коммит, из которой будут собраны демона и запущены тесты. Переменная '{tracker_issue}' подставляется автоматически."
        )

        daemon_list = parameters.List(
            "Список демонов", required=True,
            description="Демона Движка Метрики, сборка и тестирование которых будет осуществлено."
        )

        _binary = hide(binary_task.binary_release_parameters_list(stable=True))

    def on_save(self):
        self.release.author = self.real_author

        if not self.Parameters.arcadia_url:
            self.Parameters.arcadia_url = "arcadia-arc:/#users/{}/{{tracker_issue}}".format(self.release.author)

        if self.Parameters.tracker_issue:
            self.Parameters.arcadia_url = self.Parameters.arcadia_url.format(tracker_issue=self.Parameters.tracker_issue)

        release_helper.ReleaseHelper.check_daemons(self, self.Parameters.daemon_list)

    def on_enqueue(self):
        self.release.arcadia_url = self.Parameters.arcadia_url
        self.release.version = "2.{}".format(self.id)
        self.release.daemons = dict((daemon, self.release.version) for daemon in self.Parameters.daemon_list)

        release_helper.ReleaseHelper.update_description(self, self.release.author, self.Parameters.daemon_list, self.release.arcadia_url)

        release_helper.ReleaseHelper.set_build_description(self)

        release_helper.ReleaseHelper.update_tests_description(self, self.Context.run_test_tasks_ids)

    def create_stages(self):
        return [
            (self.stage_create_release_issue, "Релизный тикет"),
            (self.stage_build_packages, "Сборка"),
            (self.stage_run_tests, "Тесты")
        ]

    def stage_create_release_issue(self):
        self.Context.release_issue_key = release_helper.ReleaseHelper.create_release_issue(self, self.release, self.Parameters.tracker_issue)

        release_helper.ReleaseHelper.set_create_release_issue_info(self, self.release)

    def stage_build_packages(self):
        build_tasks = release_helper.ReleaseHelper.get_build_tasks(
            self, self.release.author, self.Parameters.daemon_list, self.release.version, self.release.arcadia_url
        )

        self.run_subtasks(build_tasks, subtasks_variable=self.Context.build_packages_task_ids, info=False)

        release_helper.ReleaseHelper.set_build_info(self, self.Context.build_packages_task_ids, repo="metrika-trusty")

        release_helper.ReleaseHelper.update_build_description(self, self.Context.build_packages_task_ids)
        release_helper.ReleaseHelper.set_build_description(self)

    def stage_run_tests(self):
        self.release.tests = release_helper.ReleaseHelper.map_tests(self.release.daemons)
        test_tasks = release_helper.ReleaseHelper.get_all_tests_tasks(
            self.release, self.release.arcadia_url, REPORT_TTL, self.release.release_issue_key
        )
        self.run_subtasks(test_tasks, subtasks_variable=self.Context.run_test_tasks_ids, after_subtasks_enqueued=self.after_tests_enqueued, info=False)

    def after_tests_enqueued(self):
        release_helper.ReleaseHelper.set_run_tests_info(self, self.Context.run_test_tasks_ids)
        release_helper.ReleaseHelper.update_tests_description(self, self.Context.run_test_tasks_ids)

    def on_release(self, parameters):
        release_helper.ReleaseHelper.release(self, self.release, parameters.get("release_status"))
        self.Context.released = True
