# coding=utf-8
import logging

from sandbox import sdk2
from sandbox.common.types import task as task_type
from sandbox.projects.common.build.ya_package_config.consts import PackageType
from sandbox.projects.metrika.admins import programs_list
from sandbox.projects.metrika.core.utils import metrika_core_helper
from sandbox.projects.metrika.utils import conductor, notifications, settings

logger = logging.getLogger("release-helper")


class ReleaseHelper(metrika_core_helper.MetrikaCoreHelper):

    @staticmethod
    def create_release_issue(task, release, tracker_issue):
        search_parameters = {
            "queue": "METRIKARELEASE",
            "components": 58836,
            "type": "release",
            "branch": tracker_issue
        }
        create_parameters = search_parameters.copy()
        create_parameters.update({
            "summary": "Движок: релиз {}".format(", ".join(release.daemons.keys())),
            "assignee": release.author,
            "description": "Релиз для {}".format(tracker_issue)
        })

        issue = ReleaseHelper.find_or_create_issue(task.st_client, search_parameters, create_parameters, task.id)
        logger.info("Issue {} created.".format(issue.key))
        release.release_issue_key = issue.key
        return issue.key

    @staticmethod
    def create_conductor_issue(task, release, issue_key, branch):
        log = logger.getChild('create_conductor_issue')

        conductor_client = conductor.Conductor(task)
        packages = {}
        task.release.conductor_daemons = []
        for daemon, version in release.daemons.items():
            for package in ("{}-metrika-yandex".format(daemon), daemon):
                if (
                    programs_list.PACKAGE_JSON in task.Context.all_programs.get(daemon, []) and
                    not conductor_client.package_is_deleted(package)
                ):
                    packages[package] = version
                    task.release.conductor_daemons.append(daemon)
                    break

        if not packages:
            log.debug("Nothing to create")
            return
        else:
            log.debug("Create ticket for %s" % packages)

        description = task.st_client.issues[issue_key].summary

        no_autoinstall = branch == task_type.ReleaseStatus.STABLE

        try:
            release.conductor_ticket = conductor_client.create_conductor_ticket(
                packages, issue_key,
                branch=branch,
                no_autoinstall=no_autoinstall,
                release_issue_key=release.release_issue_key,
                description=description
            )
        except Exception as exception:
            if "IGNORE_CONDUCTOR" in task.Parameters.tags:
                logging.exception("Failed to create conductor ticket due to {}".format(exception))
                release.conductor_ticket = None
            else:
                raise exception

    @staticmethod
    def release(task, release, branch):
        ReleaseHelper.create_conductor_issue(task, release, task.Parameters.tracker_issue, branch)
        ReleaseHelper.set_release_info(task, release, branch)
        if branch == task_type.ReleaseStatus.PRESTABLE or branch == task_type.ReleaseStatus.STABLE:
            ReleaseHelper.notify_release(task, release, branch)

    @staticmethod
    def notify_release(task, release, environment):
        daemons = "Выкладка <b>{}</b> в {} в рамках:".format(
            "</b>, <b>".join(release.daemons.keys()),
            "production" if environment == task_type.ReleaseStatus.STABLE else environment
        )

        issue_info = ReleaseHelper.get_issue_info(task.Parameters.tracker_issue)

        developer = ReleaseHelper.get_duty_telegram("core")
        developer_message = "{}, вот ссылки для аппрува:".format(developer)

        conductor_links = []
        if task.release.conductor_ticket:
            conductor_links.append(
                "©️ <a href=\"https://c.yandex-team.ru/tickets/{}\">{}</a>".format(
                    task.release.conductor_ticket, ", ".join(task.release.conductor_daemons)
                )
            )
        deploy_links = []
        deploy_tickets = ReleaseHelper.get_deploy_tickets(task, environment)
        if deploy_tickets:
            deploy_release_id = ReleaseHelper.get_deploy_releases(task, environment)[0]
            deploy_links.append(
                "🚀 <a href=\"https://deploy.yandex-team.ru/releases/{}\">{}</a>".format(
                    deploy_release_id, ", ".join(deploy_ticket.get("stage_id") for deploy_ticket in deploy_tickets)
                )
            )
        approve_links = "\n".join(conductor_links + deploy_links)

        message = "{}\n{}\n\n{}\n{}".format(daemons, issue_info, developer_message, approve_links)

        notifications.Notifications.telegram(task, "Release Approves", message)

    @staticmethod
    def get_issue_info(issue_key):
        from startrek_client import Startrek

        issue = Startrek(token=sdk2.Vault.data(settings.owner, settings.tracker_token), useragent=settings.owner).issues[issue_key]

        return "<a href=\"https://st.yandex-team.ru/{0}\">{0}</a> {1}".format(issue.key, issue.summary)

    @staticmethod
    def get_duty_telegram(group):
        from metrika.pylib import duty

        duty_client = duty.DutyAPI(token=sdk2.Vault.data(settings.owner, settings.duty_token))

        return "@" + duty_client.get_duty_group_extended("metrika", group).get("duty_contacts").get("telegram")

    @staticmethod
    def get_deploy_tickets(task, environment):
        import metrika.pylib.deploy.client as deploy

        deploy_client = deploy.DeployAPI(token=sdk2.Vault.data(settings.owner, settings.yp_token))

        deploy_tickets = []
        deploy_releases = ReleaseHelper.get_deploy_releases(task, environment)
        if deploy_releases:
            for deploy_release in deploy_releases:
                deploy_client.release.wait_for_deploy_tickets(deploy_release)
                deploy_release_meta = deploy_client.release.get_deploy_tickets(deploy_release, ["/meta"])
                if deploy_release_meta:
                    deploy_tickets.extend([meta[0] for meta in deploy_release_meta])

        return deploy_tickets

    @staticmethod
    def get_deploy_releases(task, environment):
        for subtask_id in task.Context.build_packages_task_ids:
            subtask = sdk2.Task[subtask_id]
            if subtask.Parameters.package_type == PackageType.TARBALL.value:
                return [deploy_release_id for deploy_release_id in subtask.Context.ya_deploy_release_ids if "-{}".format(environment) in deploy_release_id]

    @staticmethod
    def set_create_release_issue_info(task, release):
        create_release_issue_info = "Задача:<br/><a href=\"https://st.yandex-team.ru/{0}\">{0}</a>".format(release.release_issue_key)

        task.set_info(create_release_issue_info, do_escape=False)

    @staticmethod
    def set_release_info(task, release, branch):
        release_infos = []

        if release.conductor_ticket:
            template = "Кондукторный тикет в {0}:<br/><a href=\"https://c.yandex-team.ru/tickets/{1}\">{1}</a> ({2})"
            rack_roll_link = "<a href=\"https://rackroll.mtrs.yandex-team.ru/rackroll/conductor/tasks\">Rack & Roll</a>"
            release_infos.append(template.format(branch, release.conductor_ticket, rack_roll_link))

        deploy_releases_ids = ReleaseHelper.get_deploy_releases(task, branch)
        if deploy_releases_ids:
            template = "Деплойный релиз в {0}:<br/><a href=\"https://deploy.yandex-team.ru/releases/{1}\">{1}</a>"
            release_infos.append(template.format(branch, deploy_releases_ids[0]))

        task.set_info("<br/>".join(release_infos), do_escape=False)
