# -*- coding: utf-8 -*-

import logging

from sandbox.common.errors import TaskFailure
from sandbox.projects.common import binary_task
from sandbox.projects.metrika.utils import CommonParameters
from sandbox.projects.metrika.utils.parameters import hide
from sandbox.projects.metrika.utils.base_metrika_task import with_parents, BaseMetrikaTask
from sandbox.projects.metrika.utils.mixins.juggler_reporter import JugglerReporterMixin
from sandbox.sdk2 import parameters


@with_parents
class MetrikaCoreRunYQL(BaseMetrikaTask, JugglerReporterMixin):
    """
        Запуск yql запросов с juggler проверкой.
    """
    juggler_host = 'metrika-sandbox'
    juggler_service = 'metrika-core-run-yql'

    class Parameters(CommonParameters):
        description = 'Run YQL query'
        kill_timeout = 2 * 60 * 60

        query = parameters.String(
            "YQL Query",
            multiline=True,
            required=True
        )

        cluster = parameters.String(
            'Cluster',
            default='hahn',
            required=True
        )

        yql_secret = parameters.Vault(
            'Vault secret name for Ydb token',
            default=('METRIKA', 'robot-metrika-yql-token'),
            required=True
        )

        juggler_service_name = parameters.String(
            'juggler-service (for schedulers)',
            default=''
        )

        print_operation_share_url = parameters.Bool("Print operation's public url in log",
                                                    default=False,
                                                    description="This option turn's on printing of \
                                                                 the public operation link in the log. \
                                                                 Don't turn on this option \
                                                                 if operation result is secure sensitive")

        _binary = hide(binary_task.binary_release_parameters_list(stable=True))

    def on_prepare(self):
        if len(self.Parameters.juggler_service_name) > 0:
            self.juggler_service = self.Parameters.juggler_service_name

    def on_execute(self):
        logger = logging.getLogger('MetrikaCoreRunYQL')

        logger.info('Creating client')

        from yql.api.v1.client import YqlClient

        client = YqlClient(db=self.Parameters.cluster, token=self.Parameters.yql_secret.data())

        query = client.query(query=self.Parameters.query, syntax_version=1)
        query.run()

        if self.Parameters.print_operation_share_url:
            self.set_info('<a href="{share_url}">Public operation link</a>'.format(share_url=query.share_url), do_escape=False)

        query.get_results()

        if not query.is_success:
            logger.warning('\n'.join(map(str, list(query.warnings))))
            logger.error('\n'.join(map(str, list(query.errors))))
            raise TaskFailure("YQL error")
