# coding=utf-8
import itertools
import os

from sandbox.projects.common import apihelpers
from sandbox.projects.metrika.core.utils import metrika_core_tests_helper
from sandbox.projects.metrika.utils import resource_types


class SourceHelper(metrika_core_tests_helper.MetrikaCoreTestsHelper):
    SOURCE_DIRS = {
        resource_types.MetrikaCoreB2bTestsData.name: [
            "/mnt2",
        ],
        resource_types.AppmetricaCoreB2bTestsData.name: [
            "/opt/clickhouse-input",
            "/opt/mobile_reports_storage",
            "/opt/mobmet_log",
            "/opt/infected_masks",
        ],
    }

    @staticmethod
    def prepare(task, hostname):
        SourceHelper.create_users(task, hostname)
        SourceHelper.load_resources(task, hostname)
        SourceHelper.dump_yt(task, hostname)
        SourceHelper.install_environment_packages(task, hostname)
        SourceHelper.configure_zookeeper(task, hostname)
        SourceHelper.configure_mysql(task, hostname)
        SourceHelper.configure_clickhouse(task, hostname)

    @staticmethod
    def load_resources(task, hostname):
        resource_id = apihelpers.get_last_resource_with_attribute(task.Parameters.resource_type, "use", task.Parameters.resource_attribute, owner="METRIKA").id

        for dir in SourceHelper.SOURCE_DIRS[task.Parameters.resource_type]:
            SourceHelper.shell(task, ["rm", "-rf", dir], hostname)
        if task.Parameters.resource_type == resource_types.MetrikaCoreB2bTestsData.name:
            SourceHelper.load_resource(task, resource_id, "/mnt2", tar=True, hostname=hostname)
        else:
            SourceHelper.load_resource(task, resource_id, "/", tar=True, hostname=hostname)

    @staticmethod
    def dump_yt(task, hostname):
        if task.Parameters.dump_yt and task.Parameters.resource_type == resource_types.MetrikaCoreB2bTestsData.name:
            watchlog_path = "//home/metrika/b2b/web/watchlog"
            dump_path = "/mnt2/logbroker/watchlog"

            SourceHelper.dump_yt_table(
                task,
                os.path.join(watchlog_path, "bs-watch-log-1"),
                os.path.join(dump_path, "stage1"),
                "bs-watch-log",
                100000,
                hostname
            )

            SourceHelper.dump_yt_table(
                task,
                os.path.join(watchlog_path, "bs-watch-log-2"),
                os.path.join(dump_path, "stage2"),
                "bs-watch-log",
                100000,
                hostname
            )

    @staticmethod
    def dump_yt_table(task, source_table, destination_folder, file_name_prefix, chunk_size, hostname):
        import zstd

        compressor = zstd.ZstdCompressor()

        SourceHelper.shell(task, ["rm", "-rf", os.path.join(destination_folder, "*")], hostname)
        records = task.yt_client.read_table(source_table, format="dsv", raw=True)

        chunk_number = 0
        chunk = list(itertools.islice(records, chunk_size))
        while chunk:
            chunk_data = compressor.compress("\n".join(chunk))
            chunk_path = os.path.join(destination_folder, "{}-{:03d}.tskv.zst".format(file_name_prefix, chunk_number))
            SourceHelper.write_to_path(task, chunk_data, chunk_path, hostname)
            chunk_number += 1
            chunk = list(itertools.islice(records, chunk_size))

    @staticmethod
    def configure_mysql(task, hostname):
        metrika_core_tests_helper.MetrikaCoreTestsHelper.configure_mysql(task, hostname)
        SourceHelper.add_mysql_user_metrika(task, "localhost", hostname)
        SourceHelper.add_mysql_user_metrika(task, "127.0.0.1", hostname)
        SourceHelper.add_mysql_user_metrika(task, "::1", hostname)

        if task.Parameters.resource_type == resource_types.MetrikaCoreB2bTestsData.name:
            SourceHelper.script(task, "for table in /mnt2/mysql/*; do mysql -u root < ${table}; done", hostname)
        else:
            SourceHelper.script(task, "mysql -u root < /mysql.sql", hostname)

    @staticmethod
    def configure_clickhouse(task, hostname):
        SourceHelper.script(task, " ".join(["DEBIAN_FRONTEND=noninteractive", "apt-get", "purge", "-y", "clickhouse-server", "clickhouse-common-static"]), hostname)
        SourceHelper.shell(task, ["rm", "-rf", "/opt/clickhouse"], hostname)
        SourceHelper.install_cluster_packages(task, ["clickhouse-server", "clickhouse-common-static", "clickhouse-server-metrika", "clickhouse-client"], "mtgiga", hostname)

        SourceHelper.shell(task, ["mkdir", "-p", "/opt/clickhouse"], hostname)
        if task.Parameters.resource_type == resource_types.MetrikaCoreB2bTestsData.name:
            SourceHelper.script(task, "cp -rf /mnt2/clickhouse/* /opt/clickhouse", hostname)
        else:
            SourceHelper.script(task, "cp -rf /opt/clickhouse-input/* /opt/clickhouse", hostname)
        SourceHelper.shell(task, ["chown", "-R", "clickhouse:clickhouse", "/opt/clickhouse"], hostname)
        SourceHelper.shell(task, ["chmod", "-R", "a+w", "/opt/clickhouse"], hostname)

        SourceHelper.shell(task, ["service", "clickhouse-server", "restart"], hostname)
