# coding=utf-8
import calendar
import datetime
import logging
import re
import time
from itertools import groupby
from operator import itemgetter

from sandbox import sdk2
from sandbox.common.types import task
from sandbox.common.types.task import ReleaseStatus
from sandbox.projects.common import binary_task
from sandbox.projects.metrika.utils import CommonParameters
from sandbox.projects.metrika.utils import settings
from sandbox.projects.metrika.utils.base_metrika_task import with_parents, BaseMetrikaTask
from sandbox.projects.metrika.utils.mixins import juggler_reporter
from sandbox.sdk2 import parameters


@with_parents
class MetrikaCoreTrackerUpdate(BaseMetrikaTask, juggler_reporter.JugglerReporterMixin):
    """
    Обновление тикетов Трекера в очередях Движка Метрики
    """

    class Parameters(CommonParameters):
        description = "Обновление тикетов Трекера в очередях Движка Метрики"

        with parameters.Group("Секреты") as secrets_group:
            tokens_secret = parameters.YavSecret("Секрет с токенами", required=True, default=settings.yav_uuid)

            dogma_token_key = parameters.String("Ключ токена Догмы в секрете", required=True, default="dogma-token")

            conductor_token_key = parameters.String("Ключ токена Кондуктора в секрете", required=True, default="conductor-token")

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def on_execute(self):
        start_timestamp = self.get_previous_task_timestamp()
        end_timestamp = self.get_local_timestamp(datetime.datetime.now())
        queues = ["METR", "MOBMET", "METRADV"]

        self.update_description(start_timestamp, end_timestamp)

        issues_to_update = self.get_issues_to_update(start_timestamp, end_timestamp, queues)
        logging.debug('Issues to update: %s', issues_to_update)

        self.set_issues_to_update_info(issues_to_update)

        self.update_issues(issues_to_update)

    def get_previous_task_timestamp(self):
        previous_success_task = sdk2.Task.find(task_type=self.type, status=task.Status.SUCCESS).limit(1).first()
        return MetrikaCoreTrackerUpdate.get_gmt_timestamp(previous_success_task.created)

    def update_description(self, start_timestamp, end_timestamp):
        time_interval = "<b>{} - {}</b>".format(datetime.datetime.fromtimestamp(start_timestamp), datetime.datetime.fromtimestamp(end_timestamp))
        self.Parameters.description = time_interval

    def get_issues_to_update(self, start_timestamp, end_timestamp, queues):
        conductor_start_timestamp = self.fix_conductor_error(start_timestamp)
        conductor_end_timestamp = self.fix_conductor_error(end_timestamp)

        commits = self.get_metrika_arc_commits(start_timestamp, end_timestamp, queues)
        issues = self.get_updated_issues(start_timestamp, end_timestamp, queues)
        tasks_testing = self.get_metrika_tasks("testing", conductor_start_timestamp, conductor_end_timestamp)
        tasks_stable = self.get_metrika_tasks("stable", conductor_start_timestamp, conductor_end_timestamp)

        issues_in_progress = self.get_issues_in_progress(commits)
        issues_in_review = self.get_issues_in_review(issues, start_timestamp, end_timestamp)
        issues_in_testing = self.get_issues_in_environment("testing", tasks_testing, queues)
        issues_in_production = self.get_issues_in_environment("production", tasks_stable, queues)

        deploy_issues = self.get_issues_from_deploy(start_timestamp, end_timestamp, queues)

        all_issues = dict(issues_in_progress.items() + issues_in_review.items() + issues_in_testing.items() + issues_in_production.items() + deploy_issues.items())

        return self.filter_core_issues(all_issues)

    def get_issues_from_deploy(self, start_timestamp, end_timestamp, queues):
        from metrika.pylib.deploy.client import DeployAPI
        deploy_api = DeployAPI(token=sdk2.yav.Secret(settings.yav_uuid).data()['deploy-token'])
        releases = deploy_api.yp_client.select_objects(
            'release',
            filter=(
                '[/meta/author_id] = "robot-metrika-test" and '
                '[/status/progress/closed/status] = "true" and '
                '[/status/progress/end_time/seconds] >= {} and '
                '[/status/progress/end_time/seconds] <= {}'.format(start_timestamp, end_timestamp)
            ),
            selectors=['/spec/title', '/spec/docker/release_type', '/spec/sandbox/release_type']
        )
        issues_env = {}
        regexp = r".*?((?:{})-\d{{1,6}}).*?".format("|".join(queues))
        for title, docker_env, sandbox_env in releases:
            env = docker_env or sandbox_env
            if env not in [ReleaseStatus.TESTING, ReleaseStatus.STABLE]:
                continue

            if env == 'stable':
                env = 'production'
            issues = set(re.findall(regexp, str(title)))
            for issue in issues:
                issues_env[issue] = env
        logging.debug('Deploy issues: %s', issues_env)
        return issues_env

    def get_metrika_arc_commits(self, start_timestamp, end_timestamp, queues):
        dogma_indexing_timeout = 60 * 60
        return self.dogma_client.get_commits(self.dogma_client.make_query(
            datetime.datetime.fromtimestamp(start_timestamp - dogma_indexing_timeout),
            datetime.datetime.fromtimestamp(end_timestamp - dogma_indexing_timeout),
            queues
        ))

    def get_updated_issues(self, start_timestamp, end_timestamp, queues):
        query = "Queue: {} and Updated: \"{}\"..\"{}\"".format(",".join(queues), datetime.datetime.fromtimestamp(start_timestamp), datetime.datetime.fromtimestamp(end_timestamp))
        return self.st_client.issues.find(query=query)

    def get_metrika_tasks(self, branch, conductor_start_timestamp, conductor_end_timestamp):
        return self.conductor_client.tasks_filter_timestamps("metrika", branch, conductor_start_timestamp, conductor_end_timestamp)

    def get_issues_in_progress(self, commits):
        issues = set(issue for commit in commits for issue in commit.get("commit", {}).get("tickets", []))
        query = "Key: {} and Status: Backlog,Open".format(",".join(issues) if issues else "_")
        return dict((issue.key, "inProgress") for issue in self.st_client.issues.find(query=query))

    def get_issues_in_review(self, issues, start_timestamp, end_timestamp):
        return dict((issue.key, "inReview") for issue in issues if self.has_review(issue, start_timestamp, end_timestamp))

    def get_issues_in_environment(self, environment, tasks, queues):
        return dict((issue_key, environment) for task in tasks for issue_key in self.get_issues_from_task(task, queues))

    def get_issues_from_task(self, task, queues):
        regexp = r".*?((?:{})-\d{{1,6}}).*?".format("|".join(queues))
        return re.findall(regexp, self.conductor_client.task(task.get("name"))._info().get("comment"))

    def filter_core_issues(self, all_issues):
        query = "Key: {} and Type: engineTask".format(",".join(all_issues.keys()) if all_issues.keys() else "_")
        logging.debug(query)
        return dict((issue.key, all_issues.get(issue.key)) for issue in self.st_client.issues.find(query=query))

    def set_issues_to_update_info(self, issues_to_update):
        self.Parameters.description += " issues count {}".format(len(issues_to_update))

        transition_groups = dict((transition, list(issue[0] for issue in issue_group)) for transition, issue_group in groupby(sorted(issues_to_update.iteritems(), key=itemgetter(1)), itemgetter(1)))
        issues_to_update_info = "<br/>".join(
            "<b>{}:</b><br/><ul>{}</ul>".format(
                transition, "\n".join("<li><a href=\"https://st.yandex-team.ru/{0}\">{0}</a> {1}</li>".format(issue, self.st_client.issues[issue].summary) for issue in issues))
            for transition, issues in transition_groups.iteritems()
        )
        if issues_to_update_info:
            self.set_info(issues_to_update_info, do_escape=False)

    def update_issues(self, issues_to_update):
        for issue_key, transition_name in issues_to_update.iteritems():
            self.change_issue_status(issue_key, transition_name)

    def change_issue_status(self, issue_key, transition_name):
        all_transitions = self.st_client.issues[issue_key].transitions.get_all()
        transition = next((transition for transition in all_transitions if transition.id == transition_name), None)
        if transition:
            transition.execute()

    @property
    def dogma_client(self):
        from metrika.pylib import dogma
        return dogma.DogmaClient(token=self.Parameters.tokens_secret.data().get(self.Parameters.dogma_token_key))

    @property
    def conductor_client(self):
        from metrika.pylib.conductor import conductor
        return conductor.ConductorAPI(oauth_token=self.Parameters.tokens_secret.data().get(self.Parameters.conductor_token_key))

    @staticmethod
    def fix_conductor_error(timestamp):
        return timestamp - 3 * 60 * 60

    @staticmethod
    def has_review(issue, start_timestamp, end_timestamp):
        return any((MetrikaCoreTrackerUpdate.is_review(remote_link) and MetrikaCoreTrackerUpdate.created_in_period(remote_link, start_timestamp, end_timestamp)) for remote_link in issue.remotelinks)

    @staticmethod
    def is_review(remote_link):
        return remote_link.object.application.name == "Arcanum"

    @staticmethod
    def created_in_period(remote_link, start_timestamp, end_timestamp):
        link_timestamp = MetrikaCoreTrackerUpdate.get_tracker_timestamp(remote_link.createdAt)
        return (link_timestamp >= start_timestamp) and (link_timestamp <= end_timestamp)

    @staticmethod
    def get_tracker_timestamp(created_at):
        return MetrikaCoreTrackerUpdate.get_gmt_timestamp(datetime.datetime.strptime(created_at, "%Y-%m-%dT%H:%M:%S.%f+0000"))

    @staticmethod
    def get_local_timestamp(datetime_time):
        return int(time.mktime(datetime_time.timetuple()))

    @staticmethod
    def get_gmt_timestamp(datetime_time):
        return calendar.timegm(datetime_time.timetuple())
