# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common import binary_task
from sandbox.projects.common.utils import sync_resource
from sandbox.projects.metrika.utils.resource_types import MetrikaAntifraudTablesUploaderBinary, MetrikaBishopConfig
from sandbox.projects.metrika.utils import MetrikaLastReleasedResource, CommonParameters
from sandbox.projects.metrika.utils.base_metrika_task import with_parents, BaseMetrikaTask
from sandbox.projects.metrika.utils.mixins.juggler_reporter import JugglerReporterMixin
from sandbox.sdk2 import parameters
from sandbox.sdk2.helpers import ProcessLog, subprocess


@with_parents
class MetrikaUploadAntifraudTables(BaseMetrikaTask, JugglerReporterMixin):
    """
    Upload antifraud\'s key-value tables from YT to YDB
    """

    juggler_host = 'metrika-sandbox'
    juggler_service = 'antifraud-tables-uploader'

    class Requirements(BaseMetrikaTask.Requirements):
        cores = 16

    class Parameters(CommonParameters):
        description = 'Upload antifraud\'s key-value tables from YT to YDB'
        kill_timeout = 72 * 60 * 60
        subprocess_timeout = 72 * 60 * 60

        binary = MetrikaLastReleasedResource('Sandbox resource with antifraud-tables-uploader binary', resource_type=MetrikaAntifraudTablesUploaderBinary)

        yt_secret = parameters.Vault(
            'Vault secret name for YT token',
            default=('METRIKA', 'METRIKA_ANTIFRAUD_TABLES_UPLOAD_YT_TOKEN'),
            required=True,
        )
        ydb_secret = parameters.Vault(
            'Vault secret name for YDB token',
            default=('METRIKA', 'METRIKA_ANTIFRAUD_TABLES_UPLOAD_YDB_TOKEN'),
            required=True,
        )

        with parameters.Group('Bishop') as bishop_settings:
            bishop_program = parameters.String(
                'Bishop program name',
                default='antifraud-tables-uploader',
                required=True
            )
            bishop_environment = parameters.String(
                'Bishop environment name',
                default='metrika.sandbox.core.antifraud_tables_uploader.production',
                required=True
            )

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def on_execute(self):
        self.fetch_binary()
        self.fetch_config()

        self.run_upload_task(
            binary_path=self.Context.binary_path,
            config_path=self.Context.config,
            yt_token=self.Parameters.yt_secret.data(),
            ydb_token=self.Parameters.ydb_secret.data(),
        )

    def fetch_config(self):
        self.Context.config = MetrikaBishopConfig(
            self,
            'antifraud-tables-uploader config',
            'config.xml',
            program=self.Parameters.bishop_program,
            environment=self.Parameters.bishop_environment
        ).fetch(token=sdk2.yav.Secret('sec-01dsfe02nr23f4n9phx6b17662').data()['bishop'])

    def fetch_binary(self):
        logging.info("Try to sync %s" % str(self.Parameters.binary))
        self.Context.binary_path = sync_resource(self.Parameters.binary)
        if not self.Context.binary_path:
            raise TaskFailure("Binary is not available")

        logging.info('Binary fetched')

    def run_upload_task(self, binary_path, config_path, yt_token, ydb_token):
        with ProcessLog(self, logger=logging.getLogger('antifraud-tables-uploader')) as process_log:
            command = [
                binary_path,
                '--config-path', config_path,
            ]

            process = subprocess.Popen(
                command,
                stdout=process_log.stdout,
                stderr=process_log.stderr,
                env={
                    'YT_TOKEN': yt_token,
                    'YDB_TOKEN': ydb_token,
                }
            )

            exit_code = process.wait(timeout=self.Parameters.subprocess_timeout)

        if exit_code != 0:
            raise TaskFailure(
                "antifraud-tables-uploader subprocess returned {} exit code, "
                "see common.log for details".format(exit_code)
            )
