# -*- coding: utf-8 -*-

import logging
import time

from sandbox.projects.common import binary_task
from sandbox.projects.metrika.utils import CommonParameters
from sandbox.projects.metrika.utils.base_metrika_task import with_parents, BaseMetrikaTask
from sandbox.projects.metrika.utils.mixins.juggler_reporter import JugglerReporterMixin
from sandbox.sdk2 import parameters


@with_parents
class MetrikaCoreUploadMysqlDictToYt(BaseMetrikaTask, JugglerReporterMixin):
    juggler_host = 'metrika-sandbox'
    juggler_service = 'upload-mysql-dict-to-yt'

    class Parameters(CommonParameters):
        description = 'Upload MySQL dictionary to YT'
        kill_timeout = 1200

        mysql_secret = parameters.Vault(
            'Vault secret name for MySQL password',
            default=('METRIKA', 'METRIKA:metrika-dicts-mtmiscsql-mysql'),
            required=True
        )

        mysql_table = parameters.String(
            'MySQL table name',
            required=True
        )

        mysql_username = parameters.String(
            'MySQL username',
            default='metrika-dicts',
            required=True
        )

        mysql_host = parameters.String(
            'MySQL host',
            required=True
        )

        mysql_port = parameters.Integer(
            'MySQL port',
            default=3306
        )

        yt_secret = parameters.Vault(
            'Vault secret name for YT token',
            default=('METRIKA', 'robot-appmetrica-yt-token'),
            required=True
        )

        yt_cluster = parameters.String(
            'YT cluster',
            default='hahn',
            required=True
        )

        yt_path = parameters.String(
            'Path to YT export directory',
            default='//home/metricmob/export',
            required=True
        )

        yt_table_name = parameters.String(
            'YT table name',
            required=True
        )

        yt_fields = parameters.String(
            'Upload this fields to YT',
            default='*',
            required=True
        )

        yt_order_by = parameters.String(
            'Order YT table by this fields',
            default=''
        )

        juggler_service_name = parameters.String(
            'juggler-service (for schedulers)',
            default=''
        )

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def on_execute(self):
        import pymysql.cursors
        from pymysql.constants import FIELD_TYPE
        from yt.wrapper import YtClient

        logger = logging.getLogger('MetrikaCoreUploadMysqlDictToYt')
        yt_client = YtClient(proxy=self.Parameters.yt_cluster, token=self.Parameters.yt_secret.data())

        self.juggler_service = self.Parameters.juggler_service_name or 'upload-mysql-dict-to-yt'

        logger.info('Downloading from MySQL')
        mysql = pymysql.connect(
            host=self.Parameters.mysql_host,
            port=self.Parameters.mysql_port,
            user=self.Parameters.mysql_username,
            password=self.Parameters.mysql_secret.data(),
            cursorclass=pymysql.cursors.DictCursor
        )

        with mysql.cursor() as mysql_cursor:
            mysql_cursor = mysql.cursor()
            order_by_section = 'ORDER BY {}'.format(self.Parameters.yt_order_by) if self.Parameters.yt_order_by else ''
            query = 'SELECT {} FROM {} {}'.format(self.Parameters.yt_fields, self.Parameters.mysql_table, order_by_section)
            logger.info(query)
            mysql_cursor.execute(query)
            description = mysql_cursor.description
            rows = mysql_cursor.fetchall()

        temp_path = '{}/tmp_{}_{}'.format(self.Parameters.yt_path, self.Parameters.yt_table_name, int(time.time()))
        actual_path = '{}/{}'.format(self.Parameters.yt_path, self.Parameters.yt_table_name)

        def get_yt_type(desc):
            types_mapping = {
                FIELD_TYPE.TINY: 'int64',
                FIELD_TYPE.SHORT: 'int64',
                FIELD_TYPE.INT24: 'int64',
                FIELD_TYPE.LONG: 'int64',
                FIELD_TYPE.LONGLONG: 'int64',
            }
            return types_mapping.get(desc[1], 'string')

        order_by_fields = map(str.strip, str(self.Parameters.yt_order_by).split(','))
        schema = (
            [{'sort_order': 'ascending', 'name': field, 'type': get_yt_type(next(desc for desc in description if desc[0] == field))} for field in order_by_fields] +
            [{'name': desc[0], 'type': get_yt_type(desc)} for desc in description if desc[0] not in order_by_fields]
        )

        logger.info('Uploading to YT: {}, {}'.format(temp_path, actual_path))
        logger.info('Schema: {}'.format(schema))
        yt_client.create('table', temp_path, attributes={'schema': schema})
        yt_client.write_table(temp_path, rows, raw=False)
        yt_client.move(temp_path, actual_path, force=True)

        self.set_info(
            '<a href="https://yt.yandex-team.ru/{}/navigation?path={}">{}</a>'.format(self.Parameters.yt_cluster, actual_path, self.Parameters.yt_table_name),
            do_escape=False
        )
