from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.metrika.core.metrika_core_run_yql import MetrikaCoreRunYQL
from sandbox.projects.metrika.core.metrika_visits_ydb_to_yt_backup import MetrikaVisitsYdbToYtBackup
from sandbox.projects.metrika.utils import CommonParameters
from sandbox.projects.metrika.utils.base_metrika_task import BaseMetrikaTask, with_parents
from sandbox.projects.metrika.utils.mixins.juggler_reporter import JugglerReporterMixin


@with_parents
class MetrikaCoreYdbToYtBackupAndProcess(BaseMetrikaTask, JugglerReporterMixin):
    juggler_host = 'metrika-sandbox'
    juggler_service = 'metrika-core-ydb-to-yt-backup-and-process'

    class Parameters(CommonParameters):
        description = 'Backup ydb to yt and execute yql query'

        juggler_service_name = sdk2.parameters.String(
            'juggler-service (for schedulers)',
            default=''
        )

        juggler_host_name = sdk2.parameters.String(
            "juggler-host (for schedulers)",
            default="metrika-sandbox"
        )

        with sdk2.parameters.Group("Step 1. YDB to YT backup") as yt_prepare:
            yt_token = sdk2.parameters.Vault(
                "YT token from Vault",
                description='"name" or "owner:name"',
                default="METRIKA:robot-metrika-yt-token",
                required=True,
            )

            yt_proxy = sdk2.parameters.String(
                "YT proxy (cluster)",
                default="hahn",
                required=True,
            )

            backup_destination_path = sdk2.parameters.String(
                "Path to the dir where backup dir will be saved",
                description="There must not be a slash in the end of destination!"
                            " In mentioned directory the new directory with name"
                            " represented by timestamp as name will be created",
                default_value="//home/metrika/cross_device/id_to_crypta_id_backup/testing",
                required=True,
            )

            ydb_token = sdk2.parameters.Vault(
                "YDB token from Vault",
                description='"name" or "owner:name"',
                default="METRIKA:ydb_robot_token",
                required=True,
            )

            ydb_endpoint = sdk2.parameters.String(
                "YDB endpoint",
                description="host:port",
                default_value="ydb-ru-prestable.yandex.net:2135",
                required=True,
            )

            ydb_database = sdk2.parameters.String(
                "YDB database name",
                default_value="/ru-prestable/metrika/testing/visits",
                required=True,
            )

            ydb_table_name = sdk2.parameters.String(
                "YDB source table name",
                default_value="/ru-prestable/metrika/testing/visits/id_to_crypta_id",
                required=True,
            )

        with sdk2.parameters.Group("Step 2. Process created backup") as process_backup:
            yql_secret = sdk2.parameters.Vault(
                'Vault secret name for YQL token',
                default=('METRIKA', 'robot-metrika-yql-token'),
                required=True
            )

            process_query = sdk2.parameters.String(
                "YQL Query",
                multiline=True,
                required=True
            )

            yt_cluster = sdk2.parameters.String(
                'YT cluster',
                default='hahn',
                required=True
            )

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def on_prepare(self):
        if self.Parameters.juggler_host_name:
            self.juggler_host = self.Parameters.juggler_host_name
        if self.Parameters.juggler_service_name:
            self.juggler_service = self.Parameters.juggler_service_name

    def on_execute(self):
        self.run_subtasks(
            (
                MetrikaVisitsYdbToYtBackup,
                dict(
                    description="{} table backup for task {}".format(self.Parameters.ydb_table_name, self.id),
                    owner=self.owner,
                    yt_token=self.Parameters.yt_token,
                    proxy=self.Parameters.yt_proxy,
                    destination_path=self.Parameters.backup_destination_path,
                    ydb_token=self.Parameters.ydb_token,
                    ydb_endpoint=self.Parameters.ydb_endpoint,
                    ydb_database=self.Parameters.ydb_database,
                    ydb_tables=[self.Parameters.ydb_table_name])
            ))[0]

        self.run_subtasks(
            (
                MetrikaCoreRunYQL,
                dict(
                    description="Process {} for task {}".format(self.id, self.Parameters.ydb_table_name),
                    owner=self.owner,
                    query=self.Parameters.process_query,
                    cluster=self.Parameters.yt_cluster,
                    yql_secret=self.Parameters.yql_secret,
                    print_operation_share_url=True)
            ))[0]
