# -*- coding: UTF-8 -*-

from sandbox.projects.common import binary_task
from sandbox.projects.metrika.core import metrika_visits_ydb_to_yt_backup
from sandbox.projects.metrika.core import metrika_visits_state_statistics
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.utils import base_metrika_task

from sandbox import sdk2
from sandbox.projects.metrika.utils.mixins import juggler_reporter

import sandbox.common.types.task as ctt


@base_metrika_task.with_parents
class MetrikaVisitsBackupAndStatistics(base_metrika_task.BaseMetrikaTask, juggler_reporter.JugglerReporterMixin):
    DEFAULT_WAIT_STATUSES = ctt.Status.Group.FINISH

    class Parameters(utils.CommonParameters):
        with sdk2.parameters.Group("Step 1.1. YT Prepare") as yt_prepare:
            yt_token = sdk2.parameters.Vault(
                "YT token from Vault",
                description='"name" or "owner:name"',
                required=True,
            )

            yt_proxy = sdk2.parameters.String(
                "YT proxy (cluster)",
                default="hahn",
                required=True,
            )

            destination_path = sdk2.parameters.String(
                "Path to the dir where backup dir will be saved",
                description="There must not be a slash in the end of destination!"
                            " In mentioned directory the new directory with name"
                            " represented by timestamp as name will be created",
                default_value="//home/metrika/disaev/ydb_backup",
                required=True,
            )

            min_backup_count = sdk2.parameters.Integer(
                "Minimum count of backup dirs in YT destination path",
                default=1,
                required=False,
            )

        with sdk2.parameters.Group("Step 1.2. YDB Prepare") as ydb_prepare:
            ydb_token = sdk2.parameters.Vault(
                "YDB token from Vault",
                description='"name" or "owner:name"',
                required=True,
            )

            ydb_endpoint = sdk2.parameters.String(
                "YDB endpoint",
                description="host:port",
                default_value="ydb-ru-prestable.yandex.net:2135",
                required=True,
            )

            ydb_database = sdk2.parameters.String(
                "YDB database name",
                default_value="/ru-prestable/metrika/testing/visits",
                required=True,
            )

            ydb_tables = sdk2.parameters.List(
                "Path to a table in ydb to be backed up. Can end with a wildcard *",
                description="Path to a table in ydb, can end with a *",
                required=True,
            )

            check_interval_time = sdk2.parameters.Integer(
                "Check progress status interval time (sec.)",
                default=10,
                required=False,
            )

        with sdk2.parameters.Group("Step 1.3. Juggler Notifications") as juggler_prepare:
            juggler_host_name = sdk2.parameters.String(
                "juggler-host (for schedulers)",
                default="metrika-sandbox"
            )
            juggler_service_name = sdk2.parameters.String(
                "juggler-service (for schedulers)",
                default="metrika-visits4d-ydb-to-yt-backup"
            )

        with sdk2.parameters.Group("Step 2. State statistics prepare") as state_statistics_prepare:
            yt_statistics_path = sdk2.parameters.String(
                "YT path to the dir where statistics will be saved (there must NOT be a '/' in the end of the path!)",
                required=True
            )

            label_cluster = sdk2.parameters.String(
                "'Cluster' label for solomon",
                description="something like cluster=solid-visits4d-test",
                default="solid-visits4d-test",
                required=True,
            )

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def get_data_from_task_context(self, task_id, param_name):
        return getattr(sdk2.Task.find(id=task_id).first().Context, param_name)

    def on_prepare(self):
        if self.Parameters.juggler_host_name:
            self.juggler_host = self.Parameters.juggler_host_name
        if self.Parameters.juggler_service_name:
            self.juggler_service = self.Parameters.juggler_service_name

    def on_execute(self):
        backup_task_id = self.run_subtasks(
            (
                metrika_visits_ydb_to_yt_backup.MetrikaVisitsYdbToYtBackup,
                dict(
                    description="Runs visits backup",
                    owner=self.owner, yt_token=self.Parameters.yt_token,
                    proxy=self.Parameters.yt_proxy,
                    destination_path=self.Parameters.destination_path,
                    min_backup_count=self.Parameters.min_backup_count,
                    ydb_token=self.Parameters.ydb_token,
                    ydb_endpoint=self.Parameters.ydb_endpoint,
                    ydb_database=self.Parameters.ydb_database,
                    ydb_tables=self.Parameters.ydb_tables,
                    juggler_host_name=self.Parameters.juggler_host_name,
                    juggler_service_name=self.Parameters.juggler_service_name,
                    )
                )
            )[0]

        self.Context.backup_task_id = backup_task_id

        self.run_subtasks(
            (
                metrika_visits_state_statistics.MetrikaVisitsStateStatistics,
                dict(
                    description="Collect statistics data from visits state and push these data to the solomon",
                    owner=self.owner,
                    yt_backup_path=self.get_data_from_task_context(self.Context.backup_task_id, "yt_backup_path"),
                    yt_statistics_path=self.Parameters.yt_statistics_path,
                    label_cluster=self.Parameters.label_cluster,
                    )
                )
            )[0]
