# -*- coding: UTF-8 -*-

from sandbox.projects.common import binary_task
from sandbox.projects.metrika.core import metrika_core_run_yql
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.utils import base_metrika_task

from sandbox import sdk2

import sandbox.common.types.task as ctt

import os


@base_metrika_task.with_parents
class MetrikaVisitsStateStatistics(base_metrika_task.BaseMetrikaTask):
    DEFAULT_WAIT_STATUSES = ctt.Status.Group.FINISH
    PROJECT = "metrika"
    SERVICE = "visits4d"
    YQL_TASK_TIMEOUT = 5400  # 90 minutes
    EVENTS_COUNT_TABLE_NAME = "events_count"

    class Parameters(utils.CommonParameters):
        with sdk2.parameters.Group("YQL Prepare") as yql_request_prepare:
            yt_backup_path = sdk2.parameters.String(
                "YT path to the dir where visits backup places (there must NOT be a '/' in the end of the path!)",
                required=True,
            )

            yt_statistics_path = sdk2.parameters.String(
                "YT path to the dir where statistics will be saved (there must NOT be a '/' in the end of the path!)",
                required=True,
            )

            yt_cluster = sdk2.parameters.String('Cluster', default='hahn', required=True)

        with sdk2.parameters.Group("Solomon Prepare") as solomon_prepare:
            label_cluster = sdk2.parameters.String(
                "'Cluster' label for solomon",
                description="something like cluster=solid-visits4d-test",
                default="solid-visits4d-test",
                required=True,
            )

        with sdk2.parameters.Group("Secrets Prepare") as secrets_prepare:
            yql_secret = sdk2.parameters.Vault(
                'Vault secret name for Ydb token', default=('METRIKA', 'robot-metrika-yql-token'), required=True
            )

            yt_token = sdk2.parameters.Vault(
                "YT token from Vault",
                description='"name" or "owner:name"',
                default=('METRIKA', 'robot-metrika-yt-token'),
                required=True,
            )

            solomon_token = sdk2.parameters.Vault(
                "Solomon token from Vault",
                description='"name" or "owner:name"',
                default=('METRIKA', 'robot-metrika-solomon-token'),
                required=True,
            )

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def on_execute(self):
        self._push_events_stats_to_solomon()

    def _push_events_stats_to_solomon(self):
        from library.python import resource

        events_statistics_query = resource.find("events_request_template").format(
            table_path=self.Parameters.yt_backup_path,
            statistics_path=self.Parameters.yt_statistics_path,
            events_count_table_name=self.EVENTS_COUNT_TABLE_NAME,
        )
        self._run_yql_request("events_statistics", events_statistics_query, "Collect statistics data from events table")
        self._push_data_to_solomon(self._get_events_count_data_to_push())
        self._push_data_to_solomon(self._get_tables_count_data_to_push())

    def _run_yql_request(self, unique_query_name, query, desc):
        yql_task_id = self.run_subtasks(
            (
                metrika_core_run_yql.MetrikaCoreRunYQL,
                dict(
                    description=desc,
                    owner=self.owner,
                    query=query,
                    cluster=self.Parameters.yt_cluster,
                    yql_secret=self.Parameters.yql_secret,
                    print_operation_share_url=True,
                ),
            )
        )[0]

        setattr(self.Context, "{unique_query_name}_task_id".format(unique_query_name=unique_query_name), yql_task_id)

    def _get_events_count_data_to_push(self):
        from yt import wrapper as yt

        yt.config.set_proxy(self.Parameters.yt_cluster)
        yt.config["token"] = self.Parameters.yt_token.data()
        event_statistics_table_iterator = yt.read_table(
            "{statistics_path}/{events_count_table_name}".format(
                statistics_path=self.Parameters.yt_statistics_path, events_count_table_name=self.EVENTS_COUNT_TABLE_NAME
            ),
            format="json",
            raw=False,
        )
        data_to_push = {"metrics": []}

        while True:
            try:
                row = event_statistics_table_iterator.next()
                data_to_push["metrics"].append(
                    {
                        "labels": {
                            "host": "deploy",
                            "type": str(row["Type"]),
                            "deleted": str(row["Deleted"]),
                            "Metric": "ydb.stateStatistics.count",
                        },
                        "value": row["cnt"],
                    }
                )
            except StopIteration:
                break

        return data_to_push

    def _get_tables_count_data_to_push(self):
        from yt import wrapper as yt

        yt.config.set_proxy(self.Parameters.yt_cluster)
        yt.config["token"] = self.Parameters.yt_token.data()

        all_files_list = yt.list("{yt_backup_path}".format(yt_backup_path=self.Parameters.yt_backup_path))
        data_to_push = {"metrics": []}

        for node in all_files_list:
            node_path = os.path.join(self.Parameters.yt_backup_path, node)
            if yt.get_attribute(node_path, "type") == "table":
                spaceBytes = int(yt.get_attribute(node_path, "resource_usage")['disk_space'])

                data_to_push["metrics"].append(
                    {
                        "labels": {
                            "host": "deploy",
                            "tableName": node,
                            "Metric": "ydb.stateStatistics.table_size_bytes",
                        },
                        "value": spaceBytes,
                    }
                )

        return data_to_push

    def _push_data_to_solomon(self, data_to_push):
        from metrika.pylib.solomon import SolomonPusherOAuth

        solomon_client = SolomonPusherOAuth(
            self.PROJECT,
            self.Parameters.label_cluster,
            self.SERVICE,
            self.Parameters.solomon_token.data(),
        )
        solomon_response = solomon_client.push_json(data_to_push)

        if solomon_response.status_code != 200:
            raise Exception(
                "Push to Solomon failed. Response status code isn't 200. Current code is {solomon_response_code}; content: {content}".format(
                    solomon_response_code=solomon_response.status_code, content=solomon_response.content
                )
            )
