# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common import binary_task
from sandbox.projects.common.utils import sync_resource
from sandbox.projects.metrika.utils.resource_types import (
    MetrikaInformerDataGetterBinary, MetrikaInformerData, MetrikaBishopConfig)
from sandbox.projects.metrika.utils import MetrikaLastReleasedResource, CommonParameters
from sandbox.projects.metrika.utils.base_metrika_task import with_parents, BaseMetrikaTask
from sandbox.projects.metrika.utils.mixins.juggler_reporter import JugglerReporterMixin
from sandbox.projects.metrika.utils.mixins.dyn_resource import DynResourceMixin
from sandbox.sdk2 import parameters, ResourceData
from sandbox.sdk2.helpers import ProcessLog, subprocess

INFORMER_DATA_FILE_NAME = 'informer.data'


@with_parents
class MetrikaGetInformerData(BaseMetrikaTask, JugglerReporterMixin, DynResourceMixin):
    """
    Prepare counters traffic data for /informer handler of phantom2d server
    """

    juggler_host = None
    juggler_service = None

    class Parameters(CommonParameters):
        description = 'Prepare counters traffic data for informer'
        kill_timeout = 1200
        binary = MetrikaLastReleasedResource('Sandbox resource with informer-data-getter binary', resource_type=MetrikaInformerDataGetterBinary)
        informer_data_ttl = parameters.Integer(
            'Output informer data resource TTL in days',
            default=2,
            required=True
        )
        subprocess_timeout = parameters.Integer(
            'informer-data-getter subprocess waiting time in seconds',
            default=900,
            required=True
        )

        with parameters.Group("Secrets"):
            mysql_secret = parameters.Vault(
                'Vault secret name for MySQL password',
                default=('METRIKA', 'METRIKA_INFORMER_MYSQL_PASSWORD'),
                required=True,
            )
            mtaggr_secret = parameters.Vault(
                'Vault secret name for ClickHouse password',
                default=('METRIKA', 'METRIKA_INFORMER_MTAGGR_PASSWORD'),
                required=True,
            )
            vault_token = parameters.YavSecret(
                'Vault secret name for vault token (we need to go deeper)',
                default='sec-01dvdgt2khkskws8rzkvmyvq2w#token',
                required=True,
            )

        debug = parameters.Bool(
            'Debug',
            default=False,
            required=True,
        )

        resource_environment = parameters.String(
            'Resource environment',
            default='stable',
            required=True,
            choices=[("stable", "stable"), ("testing", "testing")],
        )

        with parameters.Group('Bishop') as bishop_settings:
            bishop_program = parameters.String(
                'Bishop program name',
                default='informer-data-getter',
                required=True
            )
            bishop_environment = parameters.String(
                'Bishop environment name',
                default='metrika.sandbox.core.informer.production',
                required=True
            )

        with parameters.Output:
            informer_data = parameters.Resource(
                'Exported informer data resource',
                resource_type=MetrikaInformerData
            )

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def get_event(self):
        return 'informer-data-getter', 'metrika-sandbox-' + self.Parameters.resource_environment, None, None

    def on_execute(self):
        self.fetch_binary()
        self.fetch_config()

        self.get_informer_data(
            binary_path=self.Context.binary_path,
            config_path=self.Context.config,
            mysql_password=self.Parameters.mysql_secret.data(),
            mtaggr_password=self.Parameters.mtaggr_secret.data(),
            vault_token=self.Parameters.vault_token.data().get(self.Parameters.vault_token.default_key),
        )

    def fetch_config(self):
        self.Context.config = MetrikaBishopConfig(
            self,
            'informer-data-getter config',
            'config.xml',
            program=self.Parameters.bishop_program,
            environment=self.Parameters.bishop_environment
        ).fetch(token=sdk2.yav.Secret('sec-01dsfe02nr23f4n9phx6b17662').data()['bishop'])

    def fetch_binary(self):
        logging.info("Try to sync %s" % str(self.Parameters.binary))
        self.Context.binary_path = sync_resource(self.Parameters.binary)
        if not self.Context.binary_path:
            raise TaskFailure("Binary is not available")

        logging.info('Binary fetched')

    def get_informer_data(self, binary_path, config_path, mysql_password, mtaggr_password, vault_token):
        self.Parameters.informer_data = MetrikaInformerData(
            self,
            'Exported informer data',
            INFORMER_DATA_FILE_NAME,
            ttl=self.Parameters.informer_data_ttl,
            environment=self.Parameters.resource_environment
        )
        resource_data = ResourceData(self.Parameters.informer_data)
        self.Context.resource_data_path = str(resource_data.path)

        with ProcessLog(self, logger=logging.getLogger('informer-data-getter')) as process_log:
            command = [
                binary_path,
                '--config-path', config_path,
                '--result-path', self.Context.resource_data_path,
            ]
            if self.Parameters.debug:
                command.append('--dump-debug-string')

            process = subprocess.Popen(
                command,
                stdout=process_log.stdout,
                stderr=process_log.stderr,
                env={
                    'METRIKA_MYSQL_PASSWORD': mysql_password,
                    'METRIKA_INFORMER_MTAGGR_PASSWORD': mtaggr_password,
                    'METRIKA_VAULT_TOKEN': vault_token,
                }
            )

            exit_code = process.wait(timeout=self.Parameters.subprocess_timeout)

        if exit_code != 0:
            resource_data.broken()

            raise TaskFailure(
                "informer-data-getter subprocess returned {} exit code, "
                "see common.log for details".format(exit_code)
            )

        resource_data.ready()

    def mark_released_resources(self, status, ttl="inf"):
        super(MetrikaGetInformerData, self).mark_released_resources(status, ttl=self.Parameters.informer_data_ttl)
