from __future__ import unicode_literals

import shutil

from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.projects import resource_types
from sandbox.projects.common import binary_task
from sandbox.projects.metrika.utils.mixins.console import BaseConsoleMixin
from sandbox.projects.metrika.utils.resource_types import METRIKA_RELEASERS
from sandbox.projects.metrika.utils import CommonParameters
from sandbox.projects.metrika.utils.base_metrika_task import with_parents, BaseMetrikaTask
from sandbox.projects.metrika.utils.mixins.juggler_reporter import JugglerReporterMixin
from sandbox.projects.metrika.utils.mixins.dyn_resource import DynResourceMixin


class RipeDBCalc(sdk2.Resource):
    name = 'RIPEDB_CALC'
    any_arch = True
    auto_backup = True
    restart_policy = ctr.RestartPolicy.DELETE
    releasable = True
    releasers = METRIKA_RELEASERS


class RipeDBAll(sdk2.Resource):
    name = 'RIPEDB_ALL'
    any_arch = True
    auto_backup = True
    restart_policy = ctr.RestartPolicy.DELETE
    releasable = True
    releasers = METRIKA_RELEASERS


CONFIG = """
<yandex>
    <logger>
        <level>trace</level>
    </logger>

    <mysql_metrica>
        <db>ripedb</db>
        <enable_local_infile>true</enable_local_infile>
        <host>{host}</host>
        <password>{password}</password>
        <port>3306</port>
        <user>metrika</user>
    </mysql_metrica>

    <path>{ripedb_dir}/ripe.db.inetnum</path>
    <out_dir>{ripedb_dir}/</out_dir>
</yandex>"""


@with_parents
class MetrikaUploadRipeDB(BaseMetrikaTask, JugglerReporterMixin, BaseConsoleMixin, DynResourceMixin):
    class Requirements(BaseMetrikaTask.Requirements):
        disk_space = 5 * 1024  # ripe.db.inetnum

    class Parameters(CommonParameters):
        host = sdk2.parameters.String('MDB host', required=True, default='sas-040f29gst0x9zijs.db.yandex.net')
        password = sdk2.parameters.YavSecret('MySql metrika pass', required=True, default='sec-01e98gq1024m7wr98rt39d3x0k')
        ripedb_updater = sdk2.parameters.Resource('ripedb-updater binary', required=True, resource_type=resource_types.OTHER_RESOURCE, default_value=1686529179)
        _binary = binary_task.binary_release_parameters_list(stable=True)

    def on_execute(self):
        ripedb_dir = self.path('ripedb-updater')
        ripedb_dir.joinpath('work-dir').mkdir(parents=True)

        config = ripedb_dir.joinpath('config.xml')
        config.write_text(
            CONFIG.format(host=self.Parameters.host, password=self.Parameters.password.data()['password'], ripedb_dir=ripedb_dir.as_posix()),
            encoding='utf8'
        )

        self._execute_script(
            'wget -O ripe.db.inetnum.gz "ftp://ftp.ripe.net/ripe/dbase/split/ripe.db.inetnum.gz" && '
            'gzip -d ripe.db.inetnum.gz && '
            'iconv -f latin1 ripe.db.inetnum -o {}/ripe.db.inetnum'.format(ripedb_dir.as_posix())
        )

        binary = sdk2.ResourceData(self.Parameters.ripedb_updater).path.as_posix()
        self._execute_shell_and_check([
            binary, '-C', config.as_posix(),
            '-L', self.log_path('ripedb_updater_stdout.log').as_posix(),
            '-E', self.log_path('ripedb_updater_stderr.log').as_posix(),
        ])

        all_dir = ripedb_dir.joinpath('ripedb')
        all_dir.mkdir()
        for file in ['Networks.txt', 'NetworksDescriptions.txt']:
            shutil.move(ripedb_dir.joinpath(file).as_posix(), all_dir.as_posix())
        ripedb_all_res = RipeDBAll(self, 'ripedb_all', all_dir, ttl=30)
        sdk2.ResourceData(ripedb_all_res).ready()

        ripedb_calc_res = RipeDBCalc(self, 'ripedb_calc', ripedb_dir.joinpath('IPToLeafNetwork.txt'), ttl=30)
        sdk2.ResourceData(ripedb_calc_res).ready()
