# coding=utf-8
import itertools
import os
import re

import json
import logging

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.utils import get_task_link
from sandbox.projects.common import constants, link_builder
from sandbox.projects.common.build.YaPackage2 import YaPackage2Parameters
from sandbox.projects.common.build.ya_package_config.consts import PackageType
from sandbox.projects.common.vcs.arc import Arc
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.admins.programs_list import CONFIG_JSON, DOCKER_JSON, PACKAGE_JSON, TARBALL_JSON
from sandbox.projects.metrika.core import appmetrica_core_arcadia_b2b_tests_run, appmetrica_core_b2b_tests_run, core_tests, metrika_core_arcadia_b2b_tests_run, \
    metrika_core_arcadia_tests_run, metrika_core_b2b_tests_run, metrika_core_local_functional_tests_run
from sandbox.projects.metrika.core.appmetrica_core_b2b_scenario_execute import tests_helper as appmetrica_tests_helper
from sandbox.projects.metrika.core.metrika_core_b2b_scenario_execute import tests_helper as metrika_tests_helper
from sandbox.projects.metrika.java.utils import metrika_java_helper
from sandbox.projects.metrika.utils import arcanum_api, notifications, resource_types, settings, vcs
from sandbox.projects.metrika.utils.task_types import PRIVATE_YT_STORE_PARAMS
from sandbox.projects.metrika.utils.task_types.ya_package import MetrikaYaPackage

logger = logging.getLogger("metrika-core-helper")


class MetrikaCoreHelper(metrika_java_helper.MetrikaJavaHelper):

    @staticmethod
    def map_tests(daemon_dictionary):
        """
        :param daemon_dictionary: словарь, ключ - наименование демона, значение - его версия
        :return: словарь, ключ - тип тестов, значение - список наименований демонов, покрытых данным типом тестов
        """
        arcanum = arcanum_api.ArcanumApi(token=sdk2.Vault.data(settings.owner, settings.arcanum_token))

        tested_daemons = {
            core_tests.CoreTests.ARCADIA_FUNCTIONAL.value: MetrikaCoreHelper._get_arcadia_tests_covered_daemons(arcanum),
            core_tests.CoreTests.APP_B2B.value: appmetrica_tests_helper.TestsHelper.get_daemons_under_arcadia_b2b(),
            core_tests.CoreTests.WEB_B2B.value: metrika_tests_helper.TestsHelper.get_daemons_under_arcadia_b2b(),
            core_tests.CoreTests.WEB_FUNCTIONAL.value: ["visits4d", "counters-server", "goals-server"]
        }

        return {test_type: list(set(daemons).intersection(daemon_dictionary.keys())) for test_type, daemons in tested_daemons.items()}

    @staticmethod
    def update_build_description(task, build_tasks_ids):
        build_description = MetrikaCoreHelper._blue('Build info:')
        with vcs.mount_arc('arcadia-arc:/#trunk') as arcadia:
            arc = Arc()
            for task_id in build_tasks_ids:
                build_task = sdk2.Task[task_id]
                task_rev = build_task.Context.revision
                if task_rev.startswith('arc:'):
                    task_rev = task_rev.split(':')[1]
                else:
                    task_rev = arc.show(arcadia, commit='r' + task_rev, as_dict=True)[0]['commits'][0]['commit']
                revision_info = arc.show(arcadia, commit=task_rev, as_dict=True)[0]['commits'][0]
                logging.debug('Revision info: %s', revision_info)
                svn_revision = revision_info.get('revision')
                arc_commit = revision_info.get('commit')
                if not svn_revision:
                    # arcadia_url is arc commit
                    trunk_revision = arc.get_merge_base(arcadia, branch=task_rev)
                    logging.debug('Merge base: %s', trunk_revision)
                    trunk_revision_info = arc.show(arcadia, commit=trunk_revision, as_dict=True)[0]['commits'][0]
                    logging.debug('Merge revision info: %s', trunk_revision_info)
                    svn_revision = svn_revision or trunk_revision_info.get('revision')
                    if not svn_revision:
                        # merged, merge-base == arcadia_url, but there is no info about svn_revision
                        commits = arc.log(arcadia, start_commit=trunk_revision, end_commit='', author=build_task.author, as_dict=True)
                        logging.debug(commits)
                        svn_revision = commits[-1]['revision']

                build_description += '<br/>'.join([
                    '',
                    '{}'.format(MetrikaCoreHelper._green(build_task.Parameters.package_type.upper())),
                    'date: {}'.format(MetrikaCoreHelper._blue(revision_info['date'].replace('T', ' '))),
                    'branch point svn revision: {}'.format(MetrikaCoreHelper._blue('r' + str(svn_revision))),
                    'arc commit: {}'.format(MetrikaCoreHelper._blue(arc_commit)),
                ])

        task.Context.build_description = build_description

    @staticmethod
    def set_build_description(task):
        if task.Context.build_description:
            MetrikaCoreHelper._update_description(task, task.Context.build_description, id='builds', separator='\n')

    @staticmethod
    def update_tests_description(task, test_tasks_ids):
        if test_tasks_ids:
            tests_description = "<br/>".join(MetrikaCoreHelper._get_tests_description(task_id) for task_id in test_tasks_ids if task_id)

            MetrikaCoreHelper._update_description(task, tests_description, id="auto_tests", separator="\n")

    @staticmethod
    def get_build_tasks(task, author, daemon_list, version, arcadia_url):
        task.Context.all_programs = MetrikaCoreHelper.get_all_programs(arcadia_url, "metrika/core")

        MetrikaCoreHelper._check_buildability(task.Context.all_programs, daemon_list)

        daemons_with_packages = [daemon for daemon in daemon_list if PACKAGE_JSON in task.Context.all_programs.get(daemon)]
        package_jsons = [task.Context.all_programs.get(daemon).get(PACKAGE_JSON) for daemon in daemons_with_packages]

        daemons_with_images = [daemon for daemon in daemon_list if DOCKER_JSON in task.Context.all_programs.get(daemon)]
        docker_jsons = [task.Context.all_programs.get(daemon).get(DOCKER_JSON) for daemon in daemons_with_images]

        daemons_with_tarballs = [daemon for daemon in daemon_list if TARBALL_JSON in task.Context.all_programs.get(daemon)]
        tarball_jsons = [task.Context.all_programs.get(daemon).get(TARBALL_JSON) for daemon in daemons_with_tarballs]

        daemons_with_configs = [daemon for daemon in daemon_list if CONFIG_JSON in task.Context.all_programs.get(daemon)]
        config_jsons = [task.Context.all_programs.get(daemon).get(CONFIG_JSON) for daemon in daemons_with_configs]

        build_tasks = []
        common_params = {
            YaPackage2Parameters.checkout_arcadia_from_url: arcadia_url,
            YaPackage2Parameters.build_type: constants.PROFILE_BUILD_TYPE,
            YaPackage2Parameters.custom_version: version,
            YaPackage2Parameters.run_tests: False,
            YaPackage2Parameters.run_long_tests: False,
            YaPackage2Parameters.clear_build: True,
            YaPackage2Parameters.semi_clear_build: True,
            YaPackage2Parameters.resource_type: resource_types.BaseMetrikaBinaryResource.name,
            YaPackage2Parameters.yp_token_vault: settings.yp_token,
            YaPackage2Parameters.release_to_ya_deploy: True,
        }
        common_params.update(PRIVATE_YT_STORE_PARAMS)
        if package_jsons:
            params = {
                "description": MetrikaCoreHelper._get_description("Debian сборка демонов Движка Метрики ", author, daemons_with_packages, arcadia_url),
                YaPackage2Parameters.package_type: PackageType.DEBIAN.value,
                YaPackage2Parameters.packages: ";".join(package_jsons),
                YaPackage2Parameters.key_user: "robot-metrika-admin",
                YaPackage2Parameters.publish_package: True,
                YaPackage2Parameters.multiple_publish: True,
                YaPackage2Parameters.publish_to_mapping: dict((package_json, "metrika;metrika-trusty;metrika-common") for package_json in package_jsons),
                YaPackage2Parameters.dupload_max_attempts: 7,
            }
            build_tasks.append((MetrikaYaPackage, dict(common_params, **params)))
        if docker_jsons:
            params = {
                "description": MetrikaCoreHelper._get_description("Docker сборка демонов Движка Метрики ", author, daemons_with_images, arcadia_url),
                YaPackage2Parameters.package_type: PackageType.DOCKER.value,
                YaPackage2Parameters.packages: ";".join(docker_jsons),
                YaPackage2Parameters.docker_image_repository: "metrika/core",
                YaPackage2Parameters.docker_push_image: True,
                YaPackage2Parameters.docker_user: settings.login,
                YaPackage2Parameters.docker_token_vault_name: settings.docker_registry_token,
            }
            build_tasks.append((MetrikaYaPackage, dict(common_params, **params)))
        if tarball_jsons:
            params = {
                "description": MetrikaCoreHelper._get_description("Tarball сборка демонов Движка Метрики ", author, daemons_with_tarballs, arcadia_url),
                YaPackage2Parameters.package_type: PackageType.TARBALL.value,
                YaPackage2Parameters.packages: ";".join(tarball_jsons),
            }
            build_tasks.append((MetrikaYaPackage, dict(common_params, **params)))
        if config_jsons:
            params = {
                "description": MetrikaCoreHelper._get_description("Сборка конфигов для тестов демонов Движка Метрики ", author, daemons_with_configs, arcadia_url),
                YaPackage2Parameters.package_type: PackageType.TARBALL.value,
                YaPackage2Parameters.packages: ";".join(config_jsons),
                YaPackage2Parameters.resource_type: resource_types.BaseMetrikaConfigResource.name,
                YaPackage2Parameters.release_to_ya_deploy: False,
            }
            build_tasks.append((MetrikaYaPackage, dict(common_params, **params)))

        return build_tasks

    @staticmethod
    def get_all_tests_tasks(development, arcadia_url, report_ttl, tracker_issue=None, run_b2b_tests=True, run_functional_tests=True, fail_b2b_task_on_test_failure=False):
        return list(itertools.chain.from_iterable(
            MetrikaCoreHelper.get_test_tasks(
                tests_type, development, daemons, arcadia_url, report_ttl, tracker_issue,
                fail_b2b_task_on_test_failure=fail_b2b_task_on_test_failure
            )
            for tests_type, daemons in development.tests.iteritems() if daemons and MetrikaCoreHelper._need_to_run(tests_type, run_b2b_tests, run_functional_tests)
        ))

    @staticmethod
    def get_test_tasks(tests_type, development, daemons, arcadia_url, report_ttl, tracker_issue=None, fail_b2b_task_on_test_failure=False):
        description = "Тесты {} из {}".format(", ".join(daemons), arcadia_url)
        if tests_type == core_tests.CoreTests.ARCADIA_FUNCTIONAL.value:
            return MetrikaCoreHelper.get_functional_tests_tasks(daemons, arcadia_url, report_ttl, tracker_issue)
        if tests_type == core_tests.CoreTests.APP_B2B.value:
            return [(
                appmetrica_core_b2b_tests_run.AppMetricaCoreB2bTestsRun,
                dict(
                    description=description,
                    arcadia_url=arcadia_url,
                    test_packages=dict((daemon, development.daemons.get(daemon)) for daemon in daemons),
                    force_test_scenario=True,
                    force_stable_scenario=True,
                    report_ttl=report_ttl,
                    report_startrek=bool(tracker_issue),
                    issue_key=tracker_issue,
                    fail_task_on_test_failure=fail_b2b_task_on_test_failure
                )
            ), (
                appmetrica_core_arcadia_b2b_tests_run.AppMetricaCoreArcadiaB2bTestsRun,
                dict(
                    description=description,
                    arcadia_url=arcadia_url,
                    test_packages=dict((daemon, development.daemons.get(daemon)) for daemon in daemons),
                    force_test_scenario=True,
                    force_stable_scenario=True,
                    report_ttl=report_ttl,
                    report_startrek=bool(tracker_issue),
                    issue_key=tracker_issue,
                    fail_task_on_test_failure=fail_b2b_task_on_test_failure
                )
            )]
        if tests_type == core_tests.CoreTests.WEB_B2B.value:
            return [(
                metrika_core_b2b_tests_run.MetrikaCoreB2bTestsRun,
                dict(
                    description=description,
                    arcadia_url=arcadia_url,
                    test_packages=dict((daemon, development.daemons.get(daemon)) for daemon in daemons),
                    force_test_scenario=True,
                    force_stable_scenario=True,
                    report_ttl=report_ttl,
                    report_startrek=bool(tracker_issue),
                    issue_key=tracker_issue,
                    fail_task_on_test_failure=fail_b2b_task_on_test_failure
                )
            ), (
                metrika_core_arcadia_b2b_tests_run.MetrikaCoreArcadiaB2bTestsRun,
                dict(
                    description=description,
                    arcadia_url=arcadia_url,
                    test_packages=dict((daemon, development.daemons.get(daemon)) for daemon in daemons),
                    force_test_scenario=True,
                    force_stable_scenario=True,
                    report_ttl=report_ttl,
                    report_startrek=bool(tracker_issue),
                    issue_key=tracker_issue,
                    fail_task_on_test_failure=fail_b2b_task_on_test_failure
                )
            )]
        if tests_type == core_tests.CoreTests.WEB_FUNCTIONAL.value:
            counters_server = "counters-server"
            goals_server = "goals-server"

            additional_daemons = dict()

            if counters_server in development.daemons:
                additional_daemons.update({counters_server: development.daemons.get(counters_server)})

            if goals_server in development.daemons:
                additional_daemons.update({goals_server: development.daemons.get(goals_server)})

            def merge_dicts(a, b):
                a.update(b)
                return a

            packages_generator = lambda daemon: merge_dicts({daemon: development.daemons.get(daemon)}, additional_daemons)

            return [(
                metrika_core_local_functional_tests_run.MetrikaCoreLocalFunctionalTestsRun,
                dict(
                    description="Тесты {} из {}".format(daemon, arcadia_url),
                    vcs="arcadia",
                    arcadia_url=arcadia_url,
                    arcadia_path="metrika/core/tests",
                    packages=packages_generator(daemon),
                    tests="**.{}.*".format(daemon),
                    fail_task_on_test_failure=True,
                    report_ttl=report_ttl,
                    report_startrek=bool(tracker_issue),
                    issue_key=tracker_issue
                )
            ) for daemon in (["visits4d"] if counters_server in development.daemons or goals_server in development.daemons else daemons)]

    @staticmethod
    def get_functional_tests_tasks(daemons, arcadia_url, report_ttl, tracker_issue=None):
        daemon_targets = MetrikaCoreHelper._get_daemon_targets(daemons)
        tests = []

        with vcs.mount_arc(arcadia_url) as arcadia:
            for target in daemon_targets:
                target = "metrika/core/programs/{}/tests/functional".format(target)
                with open(os.path.join(arcadia, target, 'ya.make')) as yamake:
                    yamake_text = yamake.read()
                    if 'PY3TEST' not in yamake_text:
                        match = re.search(r'RECURSE\((?P<subtests>(\s+(.*?)\n)+)\)', yamake_text)
                        if match:
                            targets = []
                            for subtest in match.group('subtests').split():
                                with open(os.path.join(arcadia, target, subtest, 'ya.make')) as sub_yamake:
                                    if 'PY3TEST' in sub_yamake.read():
                                        targets.append(os.path.join(target, subtest))
                        else:
                            targets = [target]
                    else:
                        targets = [target]

                for target in targets:
                    tests.append(
                        (
                            metrika_core_arcadia_tests_run.MetrikaCoreArcadiaTestsRun,
                            dict(
                                description="Тесты {} из {}".format(target, arcadia_url),
                                checkout_arcadia_from_url=arcadia_url,
                                targets=target,
                                allure_report_ttl=report_ttl,
                                fail_task_on_test_failure=False,
                                report_startrek=bool(tracker_issue),
                                issue_key=tracker_issue
                            )
                        )
                    )
        return tests

    @staticmethod
    def all_tests_passed(test_tasks_ids):
        return all(sdk2.Task[test_task_id].is_all_tests_passed for test_task_id in test_tasks_ids)

    @staticmethod
    def send_telegram_notifications(task, recipients, test_tasks_ids):
        recipients = recipients if any(recipient for recipient in recipients) else task.real_author

        test_links = dict(
            (sdk2.Task[test_task_id].type.name,
             link_builder.HREF_TO_ITEM.format(link=get_task_link(test_task_id), name="#" + str(test_task_id)) +
             " <b>{}</b>".format("тесты пройдены" if sdk2.Task[test_task_id].is_all_tests_passed else "тесты не пройдены")) for test_task_id in test_tasks_ids
        )

        telegram_message = "\n".join(
            "<b>{}</b> {}".format(name, utils.encode(link)) for name, link in test_links.iteritems())

        notifications.Notifications.telegram(task, recipients, telegram_message)

    @staticmethod
    def send_email_notifications(task, recipients, test_tasks_ids):
        recipients = recipients if any(recipient for recipient in recipients) else task.real_author

        test_results = dict((sdk2.Task[test_task_id].type.name, sdk2.Task[test_task_id].get_html_comment()) for test_task_id in test_tasks_ids)

        email_subject = "{} {}".format(task.type.name, "#" + str(task.id))

        email_body = "<br/>".join(
            "<b>{}</b> {}<br/><hr/>".format(name, utils.encode(report)) for name, report in test_results.iteritems())

        notifications.Notifications.email(task, recipients, email_subject, email_body)

    @staticmethod
    def set_send_notifications_info(task, telegram, email):
        send_notifications_info = "Уведомления:"
        if telegram:
            send_notifications_info += "<br/>telegram"
        if email:
            send_notifications_info += "<br/>e-mail"

        task.set_info(send_notifications_info, do_escape=False)

    @staticmethod
    def _get_daemon_targets(daemons):
        arcanum = arcanum_api.ArcanumApi(token=sdk2.Vault.data(settings.owner, settings.arcanum_token))

        def map_to_proto(daemon):
            proto_daemon = MetrikaCoreHelper._get_proto_for_daemon(arcanum, daemon)
            return proto_daemon if proto_daemon is not None else daemon

        return set({d: map_to_proto(d) for d in filter(lambda d: d in daemons, MetrikaCoreHelper._get_arcadia_tests_covered_daemons(arcanum))}.values())

    @staticmethod
    def _get_arcadia_tests_covered_daemons(arcanum):
        """
        Тут нужно взять все демоны у которых есть подкаталог tests/functional
        И отдельно обработать синонимы
        """
        all_daemons = [node["name"] for node in arcanum.get_nodes_raw("metrika/core/programs") if node["type"] == "dir"]
        logger.debug("Discovered all daemons:\n{}".format("\n".join(all_daemons)))
        daemons_containing_tests = MetrikaCoreHelper._get_daemons_with_arcadia_functional_tests(arcanum)
        logger.debug("Discovered daemons with functional tests:\n{}".format("\n".join(daemons_containing_tests)))

        def predicate(daemon_name):
            if daemon_name in daemons_containing_tests:
                return True
            proto_daemon = MetrikaCoreHelper._get_proto_for_daemon(arcanum, daemon_name)
            if proto_daemon and proto_daemon in daemons_containing_tests:
                return True
            return False

        covered_daemons = list(filter(predicate, all_daemons))
        logger.info("Covered daemons:\n{}".format("\n".join(covered_daemons)))
        return covered_daemons

    @staticmethod
    def _get_daemons_with_arcadia_functional_tests(arcanum):
        def predicate(node):
            if node["type"] != "dir":
                return False
            if "tests/functional" in arcanum.get_nodes("metrika/core/programs/{}".format(node["name"])):
                return True
            if "tests" in arcanum.get_nodes("metrika/core/programs/{}".format(node["name"])):
                if "functional" in arcanum.get_nodes("metrika/core/programs/{}/tests".format(node["name"])):
                    return True
            return False

        return [daemon_node["name"] for daemon_node in arcanum.get_nodes_raw("metrika/core/programs") if predicate(daemon_node)]

    @staticmethod
    def _get_proto_for_daemon(arcanum, daemon):
        """
        :return: возвращает протодемон для daemon, если daemon является синонимом, иначе None
        """

        def predicate(data_item):
            destination = data_item.get("destination", None)
            return destination and destination.get("path", None) == "/usr/bin/{}".format(daemon)

        if PACKAGE_JSON in arcanum.get_nodes("metrika/core/programs/{}".format(daemon)):
            package_definition = json.loads(arcanum.get_blob("metrika/core/programs/{}/{}".format(daemon, PACKAGE_JSON)))
            # проверяем эвристику - в списке data есть словарь, такой у которого
            #   source - type - BUILD_OUTPUT
            #   destination - path равен "/usr/bin/<daemon-name>"
            # тогда результат
            #   четвёртый компонент пути из source - path
            # иначе - None
            try:
                data_item = next(iter(filter(predicate, package_definition["data"])), None)
                if data_item:
                    source = data_item.get("source", None)
                    if source and source.get("type", None) == "BUILD_OUTPUT":
                        return next(itertools.islice(sdk2.Path(source.get("path", None)).parts, 3, None), None)
                    else:
                        return None
                else:
                    return None
            except Exception:
                logger.exception("Ошибка при определении демона-прототипа, считаем, что демон не является синонимом.")
                return None
        else:
            return None

    @staticmethod
    def _need_to_run(tests_type, run_b2b_tests, run_functional_tests):
        return (run_b2b_tests and "b2b" in tests_type) or (run_functional_tests and "functional" in tests_type) or ("arcadia" in tests_type)

    @staticmethod
    def _get_tests_description(task_id):
        try:
            if sdk2.Task[task_id].is_all_tests_passed:
                return MetrikaCoreHelper._green(MetrikaCoreHelper._get_uncolored_tests_link(task_id))
            else:
                return MetrikaCoreHelper._red(MetrikaCoreHelper._get_uncolored_tests_link(task_id))
        except errors.TaskError:
            return MetrikaCoreHelper._grey(MetrikaCoreHelper._get_uncolored_tests_link(task_id))

    @staticmethod
    def _get_uncolored_tests_link(task_id):
        return MetrikaCoreHelper._get_tests_link(task_id, "color: inherit;")

    @staticmethod
    def _get_tests_link(task_id, style=""):
        return "<a style=\"{}\" href=\"{}\">{} #{}</a>".format(style, get_task_link(task_id), sdk2.Task[task_id].type.name, task_id)
