# coding=utf-8

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.frontend import metrika_frontend_test_stand_create
from sandbox.projects.metrika.frontend.metrika_frontend_development import development_helper, development_info
from sandbox.projects.metrika.utils import base_metrika_task
from sandbox.projects.metrika.utils import parameters as metrika_parameters
from sandbox.projects.metrika.utils.pipeline import pipeline
from sandbox.sdk2 import parameters
from sandbox.sdk2.vcs import svn


@base_metrika_task.with_parents
class MetrikaFrontendDevelopment(pipeline.PipelineBaseTask):
    """
    Разработка фронтенда Метрики
    """

    @property
    def development(self):
        return development_info.DevelopmentInfo(self.Context.development_info_state)

    class Context(pipeline.PipelineBaseTask.Context):
        development_info_state = development_info.DevelopmentInfo().state
        build_task_ids = []
        create_test_stand_task_id = []
        run_tests_task_ids = []

    class Parameters(utils.CommonParameters):
        description = "Разработка фронтенда Метрики"

        service = parameters.String("Сервис", required=True, default=metrika_frontend_test_stand_create.SERVICES[0],
                                    choices=[(item, item) for item in metrika_frontend_test_stand_create.SERVICES + metrika_frontend_test_stand_create.CROWD_SERVICES],
                                    description="Сервис, который будет собран")

        arcadia_url = metrika_parameters.ArcadiaUrl("URL Аркадии", required=True, default="arcadia-arc:/#trunk", description="Ветка или коммит, из которой будет собран стенд.")

        issue_key = metrika_parameters.TrackerIssue(
            "Ключ тикета", default=None, description="Ключ тикета, в котором нужно оставить комментарий с результатами тестов"
        )

        with parameters.Group("Настройки этапа создания тестовых стендов") as create_test_stands_group:
            with service.value["metrika-frontend"]:
                metrika_custom_api = parameters.Bool("Переопределить адрес апи метрики", default=False)
                with metrika_custom_api.value[True]:
                    faced_url = parameters.String("Faced (var faced_serverUrl)")
                    webvisor_url = parameters.String("Webvisor (var webvisor_serverUrl)")

            with service.value["appmetrica-frontend"]:
                appmetrica_custom_api = parameters.Bool("Переопределить адрес апи аппметрики", default=False)
                with appmetrica_custom_api.value[True]:
                    mobmetd_url = parameters.String("Mobmetd (var hosts_mobmetHostName)")

            data_center_params = metrika_parameters.DataCenterParameters()

            use_custom_name = parameters.Bool("Использовать произвольное имя стенда", default=False, description="По умолчанию имя стенда совпадает с веткой")

            with use_custom_name.value[True]:
                custom_name = parameters.String("Произвольное имя стенда", required=True, default="", description="Имя тестового стенда")

            deploy_description = sdk2.parameters.String("Описание ревизии в тестовом стенде", default="")

        with parameters.Group("Настройки этапа тестирования") as run_tests_group:
            run_tests = parameters.Bool("Тестирование", required=True, default=True, description="Признак необходимости запуска тестов")

        with sdk2.parameters.Output:
            with sdk2.parameters.Group("Stand urls") as output_block:
                stage_name = sdk2.parameters.String("Deploy stage name")
                test_stand_url = sdk2.parameters.String("Test stand url")
                revision = sdk2.parameters.String("Task revision tag")

        _binary = metrika_parameters.hide(binary_task.binary_release_parameters_list(stable=True))

    def on_enqueue(self):
        parsed_url = svn.Arcadia.parse_url(self.Parameters.arcadia_url)
        self.Parameters.tags = parsed_url.revision
        self.Parameters.revision = parsed_url.revision

        self.development.author = self.real_author
        self.development.select_service(self.Parameters.service)
        self.development.branch = self.Parameters.arcadia_url
        if self.Parameters.use_custom_name:
            stand_name = self.Parameters.custom_name
        elif 'users/' in parsed_url.revision:
            stand_name = parsed_url.revision.split('/')[2:][0]
        else:
            stand_name = parsed_url.revision
        self.development.stand_name = stand_name.lower().replace("/", "-").replace(".", "-")
        self.development.version = str(self.id)

        development_helper.DevelopmentHelper.update_description(self, self.development.author, [self.Parameters.service], self.development.branch)

    def create_stages(self):
        return [
            (self.stage_build, "Сборка"),
            (self.stage_create_test_stand, "Создание тестового стенда"),
            (self.stage_run_tests, "Тестирование")
        ]

    def stage_build(self):
        self.run_subtasks(development_helper.DevelopmentHelper.get_build_tasks(self.development), subtasks_variable=self.Context.build_task_ids, info=False)
        development_helper.DevelopmentHelper.set_build_info(self, self.development)

    def stage_create_test_stand(self):
        self.run_subtasks([(
            metrika_frontend_test_stand_create.MetrikaFrontendTestStandCreate,
            dict(
                description="Тестовый стенд {} из ветки {}".format(self.development.service.name, self.development.branch),
                service=self.development.service.name,
                name=self.development.stand_name,
                images=dict(("metrika/frontend/{}".format(image), self.development.version) for image in self.development.service.images),
                data_center=self.Parameters.data_center,
                deploy_description=self.Parameters.deploy_description or "{} из ветки {}".format(self.development.author, self.development.branch),
                metrika_custom_api=self.Parameters.metrika_custom_api,
                faced_url=self.Parameters.faced_url,
                webvisor_url=self.Parameters.webvisor_url,
                appmetrica_custom_api=self.Parameters.appmetrica_custom_api,
                mobmetd_url=self.Parameters.mobmetd_url,
            )
        )], subtasks_variable=self.Context.create_test_stand_task_id, info=False)

        development_helper.DevelopmentHelper.set_create_test_stand_info(self, self.development)

    def stage_run_tests(self):
        if self.Parameters.run_tests:
            run_tests_tasks = development_helper.DevelopmentHelper.get_run_tests_tasks(self, self.development)
            self.run_subtasks(run_tests_tasks, subtasks_variable=self.Context.run_tests_task_ids, info=False)

            development_helper.DevelopmentHelper.set_run_tests_info(self, self.Context.run_tests_task_ids)
        else:
            self.set_info("Запуска тестов не будет.")
