# coding=utf-8
import logging
import os

from sandbox import sdk2
from sandbox.projects.common.build.ya_package_config import consts
from sandbox.projects.metrika.frontend import metrika_frontend_acceptance_tests_run
from sandbox.projects.metrika.frontend import metrika_frontend_screenshot_tests_run
from sandbox.projects.metrika.frontend import metrika_frontend_static_upload
from sandbox.projects.metrika.java.utils import metrika_java_helper
from sandbox.projects.metrika.utils import arcanum_api, resource_types, settings
from sandbox.projects.metrika.utils.task_types import ya_package

logger = logging.getLogger("development-helper")


class DevelopmentHelper(metrika_java_helper.MetrikaJavaHelper):

    @staticmethod
    def get_build_tasks(development):
        tasks = []
        for image in development.service.images:
            build_args_dict = {
                "APP_VERSION": development.version,
                "NODE_ENV": "development",
            }
            if development.service.additional_build_variables:
                build_args_dict.update(development.service.additional_build_variables)

            tasks.append((
                ya_package.MetrikaYaPackage,
                dict(
                    description="Сборка образа {} из ветки {} для {}".format(
                        image,
                        development.branch,
                        development.service.name,
                    ),
                    checkout_arcadia_from_url=development.branch,
                    package_type=consts.PackageType.DOCKER.value,
                    packages="metrika/frontend/front/services/{}/docker.json".format(image),
                    custom_version=development.version,
                    run_tests=False,
                    run_long_tests=False,
                    clear_build=True,
                    semi_clear_build=True,
                    resource_type=resource_types.BaseMetrikaBinaryResource.name,
                    docker_build_arg=build_args_dict,
                    docker_image_repository="metrika/frontend",
                    docker_push_image=True,
                    docker_user=settings.login,
                    docker_token_vault_name=settings.docker_registry_token,
                    release_to_ya_deploy=True,
                    yp_token_vault=settings.yp_token
                )
            ))

        static_params = dict(
            description="Сборка статики из ветки {} для {}".format(development.branch, development.service.name),
            vcs="arcadia",
            arcadia_url=development.branch,
            arcadia_path="metrika/frontend/front",
            version=development.version,
            bucket=development.service.bucket,
            versioned_folders=development.service.s3_folders,
            unversioned_folders=development.service.s3_freeze_folders,
            path_to_dockerfile=development.service.node_version_from,
            env_vars={"YENV": "development", "NODE_ENV": "development", "FREEZE_FILES": "1"},
            enable_compression=development.service.s3_static_compression,
        )

        if development.service.additional_static_build_variables:
            static_params.get('env_vars').update(development.service.additional_static_build_variables)

        tasks.append((
            metrika_frontend_static_upload.MetrikaFrontendStaticUpload,
            static_params,
        ))

        return tasks

    @staticmethod
    def set_build_info(task, development):
        template = "<br/><a href=\"https://dockinfo.yandex-team.ru/api/docker/resolve?registryUrl=registry.yandex.net/metrika/frontend/{0}&tag={1}\">{0}:{1}</a>"

        build_info = "Образы:"
        for image in development.service.images:
            build_info += template.format(image, development.version)

        task.set_info(build_info, do_escape=False)

        static_info = "Статика:<br/><a href=\"https://{0}.s3.yandex.net/\">{0}.s3.yandex.net</a>".format(development.service.bucket)

        task.set_info(static_info, do_escape=False)

    @staticmethod
    def set_create_test_stand_info(task, development):
        task.Parameters.stage_name = "{}-autobeta-{}".format(development.service.name, development.stand_name)
        create_test_stand_info = "Стенд:<br/><a href=\"https://deploy.yandex-team.ru/project/{0}\">{0}</a>".format(task.Parameters.stage_name)

        task.set_info(create_test_stand_info, do_escape=False)

        task.Context.url = development.service.url_template.format(development.stand_name)
        task.Parameters.test_stand_url = task.Context.url
        test_stand_url = "Интерфейс:<br/><a href=\"{0}\">{0}</a>".format(task.Context.url)

        task.set_info(test_stand_url, do_escape=False)

    @staticmethod
    def get_run_tests_tasks(task, development):
        run_tests_tasks = []

        arcanum = arcanum_api.ArcanumApi(token=sdk2.Vault.data(settings.owner, settings.arcanum_token))
        path = str(os.path.join(metrika_frontend_acceptance_tests_run.TESTS_DIR, metrika_frontend_acceptance_tests_run.TESTS_SERVICES_DIR))
        service = task.Parameters.service.replace("-frontend", "")

        if service in arcanum.get_nodes(path):
            run_tests_tasks.append((
                metrika_frontend_acceptance_tests_run.MetrikaFrontendAcceptanceTestsRun,
                dict(
                    arcadia_url=task.Parameters.arcadia_url,
                    frontend_url=development.service.url_template.format(development.stand_name),
                    comment_issue=bool(task.Parameters.issue_key),
                    issue_key=task.Parameters.issue_key,
                    tested_service=service
                )
            ))

        run_tests_tasks.append((
            metrika_frontend_screenshot_tests_run.MetrikaFrontendScreenshotTestsRun,
            dict(
                arcadia_url=task.Parameters.arcadia_url,
                frontend_url=development.service.url_template.format(development.stand_name),
                comment_issue=bool(task.Parameters.issue_key),
                issue_key=task.Parameters.issue_key,
                service=service,
                pack="full"
            )
        ))

        return run_tests_tasks
