# coding=utf-8
from __future__ import unicode_literals

from sandbox import sdk2
from sandbox.common import patterns
from sandbox.projects.common import binary_task
from sandbox.projects.common.ya_deploy import release_integration, yp_client
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.frontend import metrika_frontend_test_stand_create
from sandbox.projects.metrika.frontend.metrika_frontend_release import release_helper, release_info
from sandbox.projects.metrika.utils import base_metrika_task, parameters, settings
from sandbox.projects.metrika.utils.mixins import releasable
from sandbox.projects.metrika.utils.parameters import hide
from sandbox.projects.metrika.utils.pipeline import pipeline
from sandbox.sdk2 import yav


@base_metrika_task.with_parents
class MetrikaFrontendRelease(pipeline.PipelineBaseTask, releasable.ReleasableTaskMixin):
    """
    Готовит релиз фронтенда Метрики
    """

    @patterns.singleton_property
    def release(self):
        self.Context.release = release_info.ReleaseInfo.fill(self.Context.release)
        return self.Context.release

    class Context(pipeline.PipelineBaseTask.Context):
        release = {}

        build_task_ids = []
        create_test_stand_task_id = []
        run_tests_task_ids = []

    class Parameters(utils.CommonParameters):
        description = "Релиз фронтенда Метрики"

        container = parameters.LastPeasantContainerResource("Environment container resource", required=True)

        service = sdk2.parameters.String(
            "Сервис", required=True, default=release_info.ReleaseInfo.SERVICES.keys()[0], choices=[(key, key) for key in release_info.ReleaseInfo.SERVICES.keys()],
            description="Сервис, релиз которого будет осуществлен."
        )

        is_new = sdk2.parameters.Bool(
            "Новый релиз", required=True, default=True,
            description="Начать ли новый релиз. Новый релиз отводится от develop. Старый продолжается в существующей ветке."
        )

        previous_release_version = sdk2.parameters.String(
            "Версия предыдущего релиза", required=True, default="",
            description="Полная версия предыдущего релиза указанного сервиса."
        )

        _binary = hide(binary_task.binary_release_parameters_list(stable=True))

    @property
    def yp_client(self):
        return yp_client.YpClient(release_integration.DEFAULT_YP_API_URL, yav.Secret(settings.yav_uuid).data()["deploy-token"])

    def on_enqueue(self):
        self.release.author = self.real_author
        self.release.previous_release_version = self.Parameters.previous_release_version
        self.release.release_version = (
            release_helper.ReleaseHelper.build_version(2, self.id, 1)
            if self.Parameters.is_new else
            release_helper.ReleaseHelper.increase_version(self.release.previous_release_version, False)
        )
        release_helper.ReleaseHelper.update_description(self, self.release.author, [self.Parameters.service], self.release.release_version)

        self.release.select_service(self.Parameters.service)

    def create_stages(self):
        return [
            (self.stage_get_release_branch, "Релизный бранч"),
            (self.stage_get_changelog, "Лог изменений"),
            (self.stage_create_release_issue, "Релизный тикет"),
            (self.stage_build_and_static, "Сборка и загрузка статики"),
            (self.stage_create_test_stand, "Создание тестового стенда"),
            (self.stage_run_tests, "Тестирование")
        ]

    def stage_get_release_branch(self):
        branch = release_helper.ReleaseHelper.get_release_branch(
            self.release.service,
            self.release.release_version if self.Parameters.is_new else self.release.previous_release_version
        )

        release_helper.ReleaseHelper.set_create_release_branch_info(self, branch)

        release_helper.ReleaseHelper.create_release_branch(self, branch)

    def stage_get_changelog(self):
        release_helper.ReleaseHelper.set_changelog_info(self)

        release_helper.ReleaseHelper.filter_changelog(self.release, self.st_client)

    def stage_create_release_issue(self):
        self.Context.release_issue_key = release_helper.ReleaseHelper.create_release_issue(self, self.release)

        release_helper.ReleaseHelper.set_create_release_issue_info(self, self.release)

    def stage_build_and_static(self):
        self.run_subtasks(release_helper.ReleaseHelper.get_build_and_static_tasks(self.release), subtasks_variable=self.Context.build_task_ids, info=False)

        release_helper.ReleaseHelper.set_build_and_static_info(self, self.release)

    def on_release(self, parameters):
        release_helper.ReleaseHelper.release_to_deploy(self, self.release, parameters)

    def stage_create_test_stand(self):
        dockers = [self.release.service.docker]
        if self.release.service.extra_dockers:
            dockers.extend(self.release.service.extra_dockers)

        self.run_subtasks([(
            metrika_frontend_test_stand_create.MetrikaFrontendTestStandCreate,
            dict(
                description="Тестовый стенд релиза",
                service=self.release.service.name + "-frontend",
                name="release",
                images=dict(("metrika/frontend/" + docker.name, self.release.release_version) for docker in dockers)
            )
        )], subtasks_variable=self.Context.create_test_stand_task_id, info=False)

        release_helper.ReleaseHelper.set_create_test_stand_info(self, self.release)

    def stage_run_tests(self):
        run_tests_tasks = release_helper.ReleaseHelper.get_run_tests_tasks(self, self.release)
        self.run_subtasks(run_tests_tasks, subtasks_variable=self.Context.run_tests_task_ids, info=False)

        release_helper.ReleaseHelper.set_run_tests_info(self, self.Context.run_tests_task_ids)
