# coding=utf-8
from __future__ import unicode_literals

import logging
import os
import re
from distutils import version

from sandbox import sdk2  # noqa
from sandbox.projects.common.build.ya_package_config import consts
from sandbox.projects.common.vcs import arc
from sandbox.projects.common.ya_deploy import release_integration
from sandbox.projects.metrika.frontend import metrika_frontend_acceptance_tests_run, metrika_frontend_static_upload, utils
from sandbox.projects.metrika.frontend.metrika_frontend_release import release_info, service_release_info  # noqa; noqa
from sandbox.projects.metrika.java.metrika_java_release import release_helper
from sandbox.projects.metrika.utils import arcanum_api, resource_types, settings, vcs
from sandbox.projects.metrika.utils.task_types import ya_package
from sandbox.sdk2.vcs import svn

logger = logging.getLogger("release-helper")


class ReleaseHelper(release_helper.ReleaseHelper):

    @staticmethod
    def create_release_branch(task, branch):
        arc_client = arc.Arc()

        service = task.release.service
        release_version = task.release.release_version
        previous_release_version = task.release.previous_release_version
        release_branch = ReleaseHelper.get_release_branch(service, release_version)
        start_point_url = ReleaseHelper.get_arcadia_url("trunk" if task.Parameters.is_new else release_branch)
        release_tag = ReleaseHelper.get_release_tag(service, release_version)
        previous_release_tag = ReleaseHelper.get_release_tag(service, previous_release_version)

        with vcs.mount_arc(start_point_url) as arcadia:
            if task.Parameters.is_new:
                arc_client.checkout(arcadia, release_branch, True)
                arc_client.push(arcadia, release_branch)

            arc_client.push(arcadia, refspecs=[("HEAD", release_tag)])

        with vcs.mount_arc(ReleaseHelper.get_arcadia_url("trunk")) as arcadia:
            arc_client.fetch(arcadia, previous_release_tag)
            arc_client.fetch(arcadia, release_tag)
            task.release.changelog = ReleaseHelper.get_changelog(arcadia, previous_release_tag, release_tag)

    @staticmethod
    def get_changelog(arcadia, previous_release_tag, release_tag):
        arc_client = arc.Arc()
        return arc_client.log(arcadia, path="metrika/frontend/front", start_commit=previous_release_tag, end_commit=release_tag)

    @staticmethod
    def get_arcadia_url(revision):
        return "{}:/#{}".format(svn.Arcadia.ARCADIA_ARC_SCHEME, revision)

    @staticmethod
    def filter_changelog(release, st_client):
        # type: (release_info.ReleaseInfo) -> None

        release.short_log = list(set(re.findall(r".*?([A-Z]+-\d+).*?", release.changelog)))

        if release.short_log:
            search_by_queue_parameters = {
                "key": release.short_log,
                "queue": release.service.release_issue.queue
            }
            search_by_tags_parameters = {
                "key": release.short_log,
                "tags": [release.service.name, "common"] + release.service.changelog_extra_tags
            }

            release.linked_log = list(set(issue.key for issue in list(st_client.issues.find(filter=search_by_queue_parameters)) + list(st_client.issues.find(filter=search_by_tags_parameters))))

    @staticmethod
    def get_release_branch(service, full_version):
        # type: (service_release_info.ServiceReleaseInfo, unicode) -> unicode

        return "releases/metrika/frontend/release-{}-{}".format(service.name, ReleaseHelper.get_release_main_version(full_version))

    @staticmethod
    def get_release_tag(service, full_version):
        # type: (service_release_info.ServiceReleaseInfo, unicode) -> unicode

        return "tags/releases/metrika/frontend/release-{}-{}".format(service.name, full_version)

    @staticmethod
    def get_release_main_version(full_version):
        # type: (unicode) -> unicode

        major, minor, patch = version.StrictVersion(full_version).version
        return "{}.{}".format(major, minor)

    @staticmethod
    def sources_dir(task):
        # type: (sdk2.Task) -> unicode

        return str(task.path("sources"))

    @staticmethod
    def create_release_issue(task, release):
        # type: (sdk2.Task, release_info.ReleaseInfo) -> None
        search_parameters = {
            "queue": release.service.release_issue.queue,
            "components": release.service.release_issue.component,
            "tags": ReleaseHelper.get_release_branch(release.service, release.release_version),
            "type": "release"
        }
        create_parameters = search_parameters.copy()
        create_parameters.update({
            "summary": "Релиз {} {}".format(release.service.name, ReleaseHelper.get_release_main_version(release.release_version)),
            "assignee": release.author,
            "followers": release.service.release_issue.followers
        })

        issue = ReleaseHelper.find_or_create_issue(task.st_client, search_parameters, create_parameters, task.id, sprint_board=release.service.release_issue.board)
        logger.info("Issue {} created.".format(issue.key))
        release.service.release_issue.key = issue.key

        logs = "\n".join([
            "<{{**Связанные тикеты**\n{}\n}}>".format("\n".join(release.linked_log) if release.linked_log else "Отсутствуют"),
            "<{{**Все тикеты**\n{}\n}}>".format("\n".join("st:" + issue for issue in release.short_log) if release.short_log else "Отсутствуют"),
            "<{{**Лог**\n%%\n{}\n%%\n}}>".format(release.changelog or "Не содержит коммитов")
        ])

        issue.comments.create(text="**!!(green)Версия {}!!**\n\n{}".format(release.release_version, logs))

        subtask_search_parameters = search_parameters.copy()
        subtask_search_parameters.update(
            type='task',
            parent=issue.key,
            summary='Помержить релизную ветку после тестирования', description='Подробности в родительской задаче'
        )

        subtask_parameters = subtask_search_parameters.copy()
        subtask_parameters.update(
            assignee=issue.assignee.login,
        )
        return ReleaseHelper.find_or_create_issue(
            task.st_client, subtask_search_parameters, subtask_parameters, task.id, sprint_board=release.service.release_issue.board
        ).key

    @staticmethod
    def get_build_and_static_tasks(release):
        # type: (sdk2.Task, release_info.ReleaseInfo) -> List[Tuple[Type[sdk2.Task], dict]]
        release_branch = ReleaseHelper.get_release_branch(release.service, release.release_version)
        release_url = ReleaseHelper.get_arcadia_url(release_branch)
        description = "Release {} build: {{}}".format(release.release_version)
        tasks = []
        dockers = [release.service.docker]
        if release.service.extra_dockers:
            dockers.extend(release.service.extra_dockers)
        for docker in dockers:
            build_args = {
                "APP_VERSION": release.release_version,
                "NODE_ENV": "production"
            }
            build_args.update(docker.env)

            tasks.append((
                ya_package.MetrikaYaPackage,
                dict(
                    description=description.format(docker.name),
                    checkout_arcadia_from_url=release_url,
                    package_type=consts.PackageType.DOCKER.value,
                    packages="metrika/frontend/front/services/{}/docker.json".format(docker.name),
                    custom_version=release.release_version,
                    run_tests=False,
                    run_long_tests=False,
                    clear_build=True,
                    semi_clear_build=True,
                    resource_type=resource_types.BaseMetrikaBinaryResource.name,
                    registry_tags=[utils.Docker.get_full_image_url(
                        "metrika/frontend/" + docker.name,
                        release.release_version
                    )],
                    docker_build_arg=build_args,
                    docker_image_repository="metrika/frontend",
                    docker_push_image=True,
                    docker_user=settings.login,
                    docker_token_vault_name=settings.docker_registry_token,
                    release_to_ya_deploy=True,
                    yp_token_vault=settings.yp_token
                )
            ))

        tasks.append((
            metrika_frontend_static_upload.MetrikaFrontendStaticUpload,
            dict(
                description="Release {} build: {}".format(release.release_version, release.service.name),
                vcs="arcadia",
                arcadia_url=release_url,
                arcadia_path="metrika/frontend/front",
                bucket=release.service.s3.bucket_name,
                version=release.release_version,
                versioned_folders=release.service.s3.folders,
                unversioned_folders=release.service.s3.freeze_folders,
                node_version=release.service.node_version,
                path_to_dockerfile=release.service.docker.path
            )
        ))

        return tasks

    @staticmethod
    def release_to_deploy(task, release, additional_parameters):
        images = []
        for build_task_id in task.Context.build_task_ids:
            build_task = sdk2.Task[build_task_id]
            if build_task.type == ya_package.MetrikaYaPackage:
                image = release_integration.make_docker_image_from_path(build_task.Context.registry_tags[0])
                images.append(image)

        subject = "{}\n{} by {}".format(release.service.release_issue.key, release.release_version, release.author)
        additional_parameters["release_subject"] = "{}\n{}".format(subject, additional_parameters["release_subject"])

        release_integration.create_docker_release(task=task, images=images, additional_parameters=additional_parameters)

    @staticmethod
    def set_create_release_branch_info(task, branch):
        # type: (sdk2.Task, unicode) -> None

        template = "<a href=\"https://a.yandex-team.ru/arc_vcs/?rev={0}\">{0}</a>"

        create_release_branch_info = "Ветка:<br/>" + template.format(branch)

        task.set_info(create_release_branch_info, do_escape=False)

    @staticmethod
    def set_changelog_info(task):
        # type: (sdk2.Task) -> None

        template = "<a href=\"https://a.yandex-team.ru/arc_vcs/diff/metrika/frontend/front?prevRev={0}&rev={1}\">{2}</a>"
        release_tag = ReleaseHelper.get_release_tag(task.release.service, task.release.release_version)
        previous_release_tag = ReleaseHelper.get_release_tag(task.release.service, task.release.previous_release_version)
        create_release_branch_info = "Лог:<br/>" + template.format(previous_release_tag, release_tag, "{}...{}".format(task.release.release_version, task.release.previous_release_version))

        task.set_info(create_release_branch_info, do_escape=False)

    @staticmethod
    def set_create_release_issue_info(task, release):
        # type: (sdk2.Task, release_info.ReleaseInfo) -> None

        create_release_issue_info = "Задача:<br/><a href=\"https://st.yandex-team.ru/{0}\">{0}</a>".format(release.service.release_issue.key)
        task.set_info(create_release_issue_info, do_escape=False)

    @staticmethod
    def set_build_and_static_info(task, release):
        # type: (sdk2.Task, release_info.ReleaseInfo) -> None

        template = "Образ:<br/><a href=\"https://dockinfo.yandex-team.ru/api/docker/resolve?registryUrl=registry.yandex.net/metrika/frontend/{0}&tag={1}\">{0}:{1}</a>"

        build_info = template.format(release.service.docker["name"], release.release_version)

        task.set_info(build_info, do_escape=False)

        upload_static_info = "Бакет:<br/><a href=\"https://{0}.s3.yandex.net/\">{0}.s3.yandex.net</a>".format(release.service.name)

        task.set_info(upload_static_info, do_escape=False)

    @staticmethod
    def set_create_test_stand_info(task, release):
        # type: (sdk2.Task, release_info.ReleaseInfo) -> None

        stage_name = "{}-autobeta-{}".format(release.service.name + "-frontend", "release")
        create_test_stand_info = "Стенд:<br/><a href=\"https://deploy.yandex-team.ru/project/{0}\">{0}</a>".format(
            stage_name)

        task.set_info(create_test_stand_info, do_escape=False)

        task.Context.url = "https://release.dev.{}.yandex.ru".format(release.service.name)
        test_stand_url = "Интерфейс:<br/><a href=\"{0}\">{0}</a>".format(task.Context.url)

        task.set_info(test_stand_url, do_escape=False)

    @staticmethod
    def get_run_tests_tasks(task, release):
        # type: (sdk2.Task, release_info.ReleaseInfo) -> List[Tuple[Type[sdk2.Task], dict]]

        arcanum = arcanum_api.ArcanumApi(token=sdk2.Vault.data(settings.owner, settings.arcanum_token))
        path = str(os.path.join(metrika_frontend_acceptance_tests_run.TESTS_DIR,
                                metrika_frontend_acceptance_tests_run.TESTS_SERVICES_DIR))
        run_tests_tasks = []

        if task.Parameters.service in arcanum.get_nodes(path):
            run_tests_tasks.append((
                metrika_frontend_acceptance_tests_run.MetrikaFrontendAcceptanceTestsRun,
                dict(
                    frontend_url=task.Context.url,
                    comment_issue=True,
                    issue_key=task.Context.release_issue_key,
                    tested_service=release.service.name
                )
            ))

        return run_tests_tasks
