# coding=utf-8
import os
import re

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.metrika import utils as metrika_utils
from sandbox.projects.metrika.frontend import utils as frontend_utils
from sandbox.projects.metrika.utils import parameters as metrika_parameters, settings, vcs
from sandbox.projects.metrika.utils.base_metrika_task import base_metrika_task
from sandbox.projects.sandbox_ci import managers
from sandbox.sdk2 import parameters

NODE_PATTERN = r"\d+(\.\d+)*"


@base_metrika_task.with_parents
class MetrikaFrontendStaticUpload(base_metrika_task.BaseMetrikaLargeTask, frontend_utils.FnmMixin):
    """
    Задание заливает статику Метрики в S3-MDS.
    """

    class Parameters(metrika_utils.CommonParameters):
        description = "Загрузка статики Метрики в S3-MDS"

        container = metrika_parameters.LastKnightContainerResource("Контейнер с окружением", required=True)

        vcs_block = metrika_parameters.VcsParameters()

        bucket = parameters.String("Бакет", required=True, default="", description="Бакет в терминах S3-MDS.")

        version = parameters.String("Версия", required=True, default="", description="Для каждой версии создается свой подкаталог в бакете.")

        versioned_folders = parameters.Dict("Версионируемые каталоги", required=True, default={}, description="Будут копироваться как подкаталоги версии.")

        unversioned_folders = parameters.Dict("Неверсионируемые каталоги", required=True, default={}, description="Будут копироваться как подкаталоги в корне.")

        with parameters.Group("Окружение") as env_vars_group:
            env_vars = parameters.Dict("Переменные окружения", required=True,
                                       default={"YENV": "production", "NODE_ENV": "production", "FREEZE_FILES": "1"},
                                       description="Дополнительные переменные окружения")

        node_version = metrika_parameters.RegexpString("Версия Node", required=False, description="Версия nodejs", regexp=NODE_PATTERN)

        path_to_dockerfile = parameters.String('Путь до докерфайла', required=False)

        enable_compression = parameters.Bool('Сжимать статику', required=True, default=True)

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def on_execute(self):
        with sdk2.helpers.ProgressMeter("Checkout"):
            with vcs.mount_arc(self.Parameters.vcs_block.arcadia_url) as arcadia_root:
                self.arcadia_root = arcadia_root

                with sdk2.helpers.ProgressMeter("Install nodejs"):
                    self.__node()

                with sdk2.helpers.ProgressMeter("Build"):
                    self.__build()

                if self.Parameters.enable_compression:
                    with sdk2.helpers.ProgressMeter("Archive"):
                        self.__archive()

                with sdk2.helpers.ProgressMeter("Upload to S3"):
                    self.__upload_to_mds()

    def __node(self):
        node_version = self.Parameters.node_version

        if not node_version:
            if self.Parameters.path_to_dockerfile:
                dir, file = os.path.split(self.Parameters.path_to_dockerfile)
                node_version = sdk2.helpers.subprocess.check_output(
                    ['grep', '-oP', "^FROM .*node:" + NODE_PATTERN, file, '-m', '1'],
                    cwd=self.__sources_dir() + '/' + dir
                ).strip()
                node_version = re.search(NODE_PATTERN, node_version).group()

                self.set_info('Версия Node из Dockerfile: {}'.format(node_version))
            else:
                node_version = '8'
                self.set_info('Версия Node по умолчанию: {}'.format(node_version))

        self.Context.node_version = node_version
        self.install_node(self.Context.node_version)

    def __build(self):
        environment = os.environ.copy().copy()
        environment.update({k: v for k, v in self.Parameters.env_vars.iteritems()})
        environment.update({"TANKER_API_TOKEN": sdk2.Vault.data(settings.owner, settings.tanker_token)})

        self._execute_script(
            """make static-build service={} version={}""".format(self.Parameters.bucket, self.Parameters.version),
            cwd=self.__sources_dir(),
            env=environment
        )

    def __archive(self):
        for source_folder, destination_folder in self.Parameters.versioned_folders.iteritems():
            if source_folder:
                frontend_utils.zopfli_and_brotli(self.__sources_dir(source_folder))

        for source_folder, destination_folder in self.Parameters.unversioned_folders.iteritems():
            if source_folder:
                frontend_utils.zopfli_and_brotli(self.__sources_dir(source_folder))

    def __upload_to_mds(self):
        client = managers.S3Manager(sdk2.Vault.data(settings.owner, settings.s3_mds_access_key_id), sdk2.Vault.data(settings.owner, settings.s3_mds_access_secret_key), self.Parameters.bucket)

        for source_folder, destination_folder in self.Parameters.versioned_folders.iteritems():
            if source_folder:
                client.upload_dir(self.__sources_dir(source_folder), os.path.join(self.Parameters.version, destination_folder or source_folder))

        for source_folder, destination_folder in self.Parameters.unversioned_folders.iteritems():
            if source_folder:
                client.upload_dir(self.__sources_dir(source_folder), destination_folder or source_folder)

    def __sources_dir(self, *path):
        return self.__workspace_dir(os.path.join(self.arcadia_root, self.Parameters.vcs_block.arcadia_path), *path)

    def __workspace_dir(self, *path):
        return str(self.path(*path))
