# coding=utf-8
import json
import logging
import os

from sandbox import sdk2
from sandbox.common import errors, fs
from sandbox.projects import resource_types
from sandbox.projects.common import binary_task, network, utils
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.build import parameters as cbp
from sandbox.projects.common.search.components import component as components_common
from sandbox.projects.metrika import utils as metrika_utils
from sandbox.projects.metrika.frontend import utils as frontend_utils
from sandbox.projects.metrika.utils import arcanum_api, artifacts, parameters as metrika_parameters, resource_types as metrika_resource_types, vcs
from sandbox.projects.metrika.utils.base_metrika_task import base_metrika_task
from sandbox.projects.metrika.utils.mixins import console
from sandbox.sdk2 import parameters

logger = logging.getLogger(__name__)

DEFAULT_REPORT_LINKS = {
    "e2e tests": "e2e/report/index.html",
    "unit tests": "unit/coverage/index.html",
    "e2e common tests": "e2e/report/common/index.html",
    "e2e proxy tests": "e2e/report/proxy/index.html",
    "e2e smoke tests": "e2e/report/smoke/index.html",
}


@base_metrika_task.with_parents
class MetrikaWatchBuild(base_metrika_task.BaseMetrikaLargeTask, console.BaseConsoleMixin):
    """
    Задание собирает Код счетчика Метрики из Аркадии
    """

    EXTS_WITHOUT_ARCHIVE = ['gif', 'png', 'jpg', 'jpeg']

    class Context(sdk2.Context):
        artifacts_resource_id = None
        resource_id = None
        report_links = {}

    class Parameters(metrika_utils.CommonParameters):
        description = "Сборка Кода счетчика Метрики из Аркадии"

        container = metrika_parameters.LastKnightContainerResource("Environment container resource", required=True)

        checkout_arcadia_from_url = cbp.ArcadiaUrl("URL Аркадии", required=True, default="arcadia-arc:/#trunk", description="Ветка или коммит, из которой будет собран Код счетчика.")

        arcadia_patch = cbp.ArcadiaPatch()

        pull_request_id = parameters.Integer('ID пулл-реквеста, куда будет записана информация о размере бандла', required=False, group=cbp.BASE_BUILD_GROUP_NAME)

        tracker_issue = metrika_parameters.TrackerIssue()

        with parameters.Group("Окружение") as env_vars_group:
            env_vars = parameters.Dict("Переменные окружения", required=True, default={"DISABLE_DEV_SSH_TUNNEL": "1"},
                                       description="Дополнительные переменные окружения")

        with parameters.Group("Тесты") as tests_group:
            is_run_smoke_test = parameters.Bool("Запускать smoke тесты", required=True, default=True)

        with parameters.Group("Сборка") as build_group:
            is_run_build = parameters.Bool("Запускать сборку", required=True, default=True)

        with parameters.Group("Секреты") as secrets_group:
            yav_secret = parameters.String("ID секрета", required=True, default="sec-01dc2e47bg95qvz40g9jx9p5gs",
                                           description="Секрет в yav, который используется для тестов (должен быть делегирован).")

            yav_mapping = parameters.Dict("Переменные окружения", required=True, default={"SELENIUM_PASSWORD": "seleniumPassword"}, description="Маппинг переменных окружения на ключи секрета.")

        _binary = binary_task.binary_release_parameters_list(stable=True)

    @property
    def arcanum_api(self):
        return arcanum_api.ArcanumApi(token=os.environ.copy()['ARC_TOKEN'])

    def on_prepare(self):
        os.environ["CI_SYSTEM"] = 'true'

    def on_execute(self):
        # Compute bundle size only if "pull_request_id" parameter is set.
        if self.Parameters.pull_request_id:
            with vcs.mount_arc(sdk2.svn.Arcadia.trunk_url()) as arcadia_trunk:
                sources_dir = os.path.join(arcadia_trunk, 'metrika/frontend/watch')

                environment = os.environ.copy()

                with sdk2.helpers.ProgressMeter('Install'):
                    self._execute_shell_and_check(['make', 'install'], cwd=sources_dir, env=environment)

                bundle_size_trunk = self.__compute_bundle_size(sources_dir, environment)

        with vcs.mount_arc(self.Parameters.checkout_arcadia_from_url) as arcadia_root:
            if self.Parameters.arcadia_patch:
                sdk.apply_patch(self, arcadia_root, self.Parameters.arcadia_patch, self.path())
            sources_dir = os.path.join(arcadia_root, "metrika/frontend/watch")

            environment = os.environ.copy()
            environment.update({k: v for k, v in self.Parameters.env_vars.iteritems()})

            for variable, secret_key in self.Parameters.yav_mapping.iteritems():
                environment[variable] = sdk2.yav.Secret(self.Parameters.yav_secret).data()[secret_key]

            with sdk2.helpers.ProgressMeter("Install"):
                self._execute_shell_and_check(["make", "install"], cwd=sources_dir, env=environment)

            exit_code_lint = None

            with sdk2.helpers.ProgressMeter("Lint"):
                exit_code_lint = self._execute_shell(["make", "lint"], cwd=sources_dir, env=environment)

            exit_code_test = None

            environment.update(self.__get_additional_env())

            with sdk2.helpers.ProgressMeter("Tests"):
                exit_code_test = self._execute_shell(["make", "test"], cwd=sources_dir, env=environment)

            if self.Parameters.is_run_build:
                with sdk2.helpers.ProgressMeter("Build"):
                    self._execute_shell_and_check(["make", "build"], cwd=sources_dir, env=environment)

                with sdk2.helpers.ProgressMeter("Archive"):
                    frontend_utils.zopfli_and_brotli(os.path.join(sources_dir, "_build"), ignore_extensions=self.EXTS_WITHOUT_ARCHIVE)

                with sdk2.helpers.ProgressMeter("Resource"):
                    local_build_dir = self.path("_build").as_posix()
                    fs.copy_dir(os.path.join(sources_dir, "_build"), local_build_dir)

                    self.Context.resource_id = metrika_resource_types.MetrikaWatch(self, self.Parameters.description, local_build_dir).id

            exit_code_smoke_test = None
            if self.Parameters.is_run_smoke_test:
                exit_code_smoke_test = self._execute_shell(["make", "test-smoke"], cwd=sources_dir, env=environment)

            self.Context.artifacts_resource_id = artifacts.archive_artifacts(
                self, os.path.join(sources_dir, "test"), self.path("artifacts").as_posix(), resource_types.TASK_CUSTOM_LOGS,
                "e2e/report", "unit/coverage"
            )
            self.Context.report_links = self.filter_report_links(DEFAULT_REPORT_LINKS)

            any_check_dropped = exit_code_lint or exit_code_test or exit_code_smoke_test
            if self.Parameters.tracker_issue:
                if any_check_dropped:
                    color, msg = 'red', 'есть падения тестов'
                else:
                    color, msg = 'green', 'все тесты пройдены'

                if self.Parameters.tracker_issue:
                    self.st_client.issues[self.Parameters.tracker_issue].comments.create(
                        text="В задаче **(({link} {name} #{id})) !!({color}){msg}!!**".format(
                            link=self.link, name=self.type.name, id=self.id,
                            color=color, msg=msg
                        )
                    )

            if any_check_dropped:
                raise errors.TaskFailure("Ошибки в тестах. Подробности в логах.")

            if self.Parameters.pull_request_id:
                bundle_size_ref = self.__compute_bundle_size(sources_dir, environment)

                self.__write_comment_to_pr(
                    self.Parameters.pull_request_id,
                    bundle_size_trunk,
                    bundle_size_ref
                )

    def on_release(self, parameters_):
        if self.Parameters.is_run_build:
            utils.set_resource_attributes(self.Context.resource_id, {"released": "stable", "ttl": "inf"})

    def __get_additional_env(self):
        env = {
            "SELENIUM_HOST": str(network.get_my_ipv6()),
            "SELENIUM_PORT": str(components_common.try_get_free_port())
        }
        logger.info("Addition environment variables: {}".format(env))
        return env

    def filter_report_links(self, links):
        resource_path = sdk2.ResourceData(sdk2.Resource[self.Context.artifacts_resource_id]).path
        filtered_links = {}
        for report_name, report_link in links.iteritems():
            if (resource_path / report_link).exists():
                filtered_links[report_name] = report_link
        return filtered_links

    @sdk2.header()
    @metrika_utils.custom_report_logger
    def report(self):
        if self.Context.artifacts_resource_id is not None:
            template_context = {"resource_link": sdk2.Resource[self.Context.artifacts_resource_id].http_proxy,
                                "reports": self.Context.report_links}
            return metrika_utils.render("header.html", template_context)
        else:
            return None

    def __compute_bundle_size(self, sources_dir, env):
        with sdk2.helpers.ProgressMeter('Size'):
            self._execute_shell_and_check(['make', 'size'], cwd=sources_dir, env=env)

        bundle_size_json_path = os.path.join(sources_dir, 'bundleSize.json')
        if not os.path.isfile(bundle_size_json_path):
            logger.error('Не найден файл bundleSize.json')

        with open(bundle_size_json_path, 'r') as bundle_size_json_file:
            bundle_size = json.load(bundle_size_json_file)

        return bundle_size

    def __write_comment_to_pr(self, pr_id, bundle_size_trunk, bundle_size_ref):
        rendered_template = metrika_utils.render(
            "bundle_size_comment.md.jinja2",
            {
                "ref": bundle_size_ref,
                "trunk": bundle_size_trunk,
            }
        )

        comment_status = self.arcanum_api.pr_comment(pr_id, rendered_template)
        if not comment_status:
            logger.error('Не удалось создать комментарий к ПР')
