# coding=utf-8
import logging

from sandbox.common import utils as common_utils
from sandbox.projects.common import binary_task
from sandbox.projects.metrika import utils as metrika_utils
from sandbox.projects.metrika.frontend import metrika_watch_deploy, metrika_watch_npm_release, utils as frontend_utils
from sandbox.projects.metrika.utils import base_metrika_task, parameters as metrika_parameters
from sandbox.sdk2 import parameters

BUNDLE_NAMES = [
    "preprod",
    "euro-preprod",
    "prod",
    "euro",
    "yandex",
    "yandex-team",
    "new-counter-experiment",
    "assessor-experiment",
]


@base_metrika_task.with_parents
class MetrikaWatchBundleRelease(base_metrika_task.BaseMetrikaTask):
    """
    Выкладка кода счетчика Метрики
    """

    class Parameters(metrika_utils.CommonParameters):
        description = "Выкладка кода счетчика Метрики"

        tracker_issue = metrika_parameters.TrackerIssue(required=True)

        watch_resource = frontend_utils.LastReleasedWatchJSResource("Released build")

        bundle_names = parameters.CheckGroup(
            "Bundle names",
            required=True,
            default=BUNDLE_NAMES,
            choices=metrika_parameters.choices(BUNDLE_NAMES)
        )

        release_npm = parameters.Bool("Release to NPM", default=True)

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def on_execute(self):
        logging.info("Started")
        self.bundles_release()
        logging.info("Finished")

        if self.Parameters.tracker_issue:
            self.report(self.Parameters.tracker_issue, self.Parameters.watch_resource)

    def bundles_release(self):
        for bundle_name in self.Parameters.bundle_names:
            subtasks = [(
                metrika_watch_deploy.MetrikaWatchDeploy,
                dict(
                    description=self.Parameters.description.split("(")[0] + "(бандл {})".format(bundle_name),
                    owner=self.Parameters.owner,
                    bundle_name=bundle_name,
                    watch_resource=self.Parameters.watch_resource,
                    tracker_issue=self.Parameters.tracker_issue,
                )
            )]

            if bundle_name == "prod" and self.Parameters.release_npm:
                subtasks.append((
                    metrika_watch_npm_release.MetrikaWatchNpmRelease,
                    dict(
                        description="Release to NPM. Generated by %s #%s" % (str(self.type), self.id),
                        watch_resource=self.Parameters.watch_resource,
                        tracker_issue=self.Parameters.tracker_issue
                    )
                ))

            self.run_subtasks(subtasks)

    def report(self, tracker_issue, watch_resource):
        resource_link = "(({} {} #{}))".format(common_utils.get_resource_link(watch_resource.id), watch_resource.type.name, watch_resource.id)
        comment = "Из ресурса **{}** были выложены бандлы **!!(green){}!!**".format(resource_link, ", ".join(self.Parameters.bundle_names))

        self.st_client.issues[tracker_issue].comments.create(text=comment)
