# coding=utf-8
import collections
import json
import logging
import os
import shutil
import time
from email.utils import formatdate

import six

from sandbox import sdk2
from sandbox.common import errors, patterns
from sandbox.common.mds.compression import base
from sandbox.common.types import resource as ctr
from sandbox.projects.common import binary_task, utils as common_utils
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.admins.infra.api import events
from sandbox.projects.metrika.frontend import metrika_watch_build
from sandbox.projects.metrika.utils import base_metrika_task, parameters, resource_types, settings
from sandbox.projects.metrika.utils.mixins import dyn_resource

Release = collections.namedtuple('Release', [
    'datetime',
    'filename',
    'release',
    'sandbox_task_id',
    'tracker_issue',
])


class MetrikaWatchBundle(sdk2.Resource):
    any_arch = True
    auto_backup = True
    restart_policy = ctr.RestartPolicy.DELETE
    releasable = True
    releasers = resource_types.METRIKA_RELEASERS


@base_metrika_task.with_parents
class MetrikaWatchDeploy(base_metrika_task.BaseMetrikaTask, dyn_resource.DynResourceMixin):
    """
    Выкладка кода счетчика Метрики
    """

    watch_data_path = None  # Путь до ресурса с js файликами

    watch_version = None  # Версия сборки кода счетчика

    file_mapping = {}  # распаршенный meta.js для текущего релиза

    bundles_mapping = {}  # распаршенный meta.js для всех бандлов с маппингом в релизе

    features_mapping = {}  # распаршенный meta.js для всех бандлов с маппингом фичей

    released_files = set()  # файлы, которые релизятся. Для их аналогов в stable или prestable будет одновлен mtime

    released_bundles_files = set()  # файлы, которые релизятся из всех бандлов. Для них будет обновлен mtime в mtime.json

    class Parameters(utils.CommonParameters):
        description = "Выкладка кода счетчика Метрики"

        watch_resource = resource_types.LastReleasedWatchJSResource("Released build")

        tracker_issue = parameters.TrackerIssue(required=True)

        bundle_name = sdk2.parameters.String("Bundle name", required=True)

        ignore_missing_watch_version = sdk2.parameters.Bool("Ignore missing watch version in watch_resource", default=False)

        custom_settings = sdk2.parameters.Bool("Set custom settings", default=False)
        with custom_settings.value[True]:
            with sdk2.parameters.Group("YT"):
                yt_block = sdk2.parameters.Info("YT Parameters")
                yt_secret = sdk2.parameters.YavSecret("yt token", default='sec-01dsfe02nr23f4n9phx6b17662', required=True)
                yt_secret_key = sdk2.parameters.String("key from secret", default='yt_oauth_token', required=True)
                yt_locke_root = sdk2.parameters.String("root path in Locke", default="//home/metrika/infra/watchjs/", required=True)

            _binary = binary_task.binary_release_parameters_list(stable=True)

    class Context(sdk2.Context):
        release_per_resource = True

    def on_execute(self):
        infra_events = events.Events(sdk2.Vault.data(settings.owner, settings.infra_token))

        event_id = None
        try:
            event_id = self.create_infra_event(infra_events)
        except Exception:
            logging.exception("Could not create infra event")

        try:
            self._get_watch_version()
            self._get_resources_data()
            self._get_bundles()
            self._get_features()
            self.release_resource()
            self.touch_released_files()
            self.write_mtime_json()
            self.write_features_json()
            self.release_sb_bundles()
        finally:
            if event_id:
                self.finish_infra_event(infra_events, event_id)

    @patterns.singleton_property
    def yt_token(self):
        return self.Parameters.yt_secret.data().get(self.Parameters.yt_secret_key)

    @patterns.singleton_property
    def mtime(self):
        self.Context.mtime = int(time.time())
        return self.Context.mtime

    @patterns.singleton_property
    def yt_locke_client(self):
        import yt.wrapper as yw
        return yw.YtClient(proxy='locke', token=self.yt_token)

    def _get_watch_version(self):
        watch_version_attr = 'watch_version'
        try:
            self.watch_version = getattr(self.Parameters.watch_resource, watch_version_attr)
        except AttributeError:
            if not self.Parameters.ignore_missing_watch_version:
                raise errors.TaskError("watch_version was not found in self.Parameters.watch_resource resource")

    def _get_resources_data(self):
        self.watch_data_path = str(sdk2.resource.ResourceData(self.Parameters.watch_resource).path)

    def _get_features(self):
        with open(os.path.join(self.watch_data_path, 'meta.json')) as meta:
            meta_data = json.load(meta)

        bundle_param = self.Parameters.bundle_name

        for bundle in meta_data:
            bundle_name = bundle['bundle']
            if bundle_param != "all!" and bundle_param != bundle_name:
                continue
            self.features_mapping[bundle_name] = bundle
            self.features_mapping[bundle_name]['files'] = [
                file for file in self.features_mapping[bundle_name]['files']
                if file['file'] != '.npmrc'
            ]
            self.features_mapping[bundle_name]['mapping'] = {
                file_path: save_as for file_path, save_as in self.features_mapping[bundle_name]['mapping'].items()
                if file_path != '.npmrc'
            }

    def _get_bundles(self):
        with open(os.path.join(self.watch_data_path, 'meta.json')) as meta:
            meta_data = json.load(meta)

        bundle_param = self.Parameters.bundle_name

        for bundle in meta_data:
            if 'mapping' not in bundle:
                continue
            bundle_name = bundle['bundle']
            if bundle_param != "all!" and bundle_param != bundle_name:
                continue
            self.bundles_mapping[bundle_name] = {}

            for file_path, save_as in bundle['mapping'].iteritems():
                if file_path == '.npmrc':
                    continue

                file_path = os.path.join(bundle_name, file_path)
                self.bundles_mapping[bundle_name][file_path] = save_as
                archived = os.path.splitext(save_as)[1][1:].lower() not in metrika_watch_build.MetrikaWatchBuild.EXTS_WITHOUT_ARCHIVE
                if archived:
                    self.bundles_mapping[bundle_name][file_path + ".br"] = save_as + ".br"
                    self.bundles_mapping[bundle_name][file_path + ".gz"] = save_as + ".gz"

                if save_as not in self.released_bundles_files:
                    self.released_bundles_files.add(save_as)
                    if archived:
                        self.released_bundles_files.add(save_as + ".br")
                        self.released_bundles_files.add(save_as + ".gz")

    def touch_released_files(self):
        bundles = self.yt_locke_client.list(os.path.join(self.Parameters.yt_locke_root, "bundles"))

        for bundle in bundles:
            if bundle == self.Parameters.bundle_name:
                continue
            root = os.path.join(self.Parameters.yt_locke_root, "bundles", bundle)

            with self.yt_locke_client.Transaction():
                self.yt_locke_client.create('document', root, recursive=True, ignore_existing=True)
                self.yt_locke_client.lock(root)

                data = self.yt_locke_client.get(root)

                if not data:
                    data = []

                for file_info in data:
                    if file_info['name'] in self.released_files:
                        logging.info("Touch file %s in %s", file_info['name'], root)
                        file_info['mtime'] = self.mtime

                self.yt_locke_client.set(root, data)

    def write_mtime_json(self):
        root = os.path.join(self.Parameters.yt_locke_root, "misc", "mtime")

        with self.yt_locke_client.Transaction():
            self.yt_locke_client.create('document', root, recursive=True, ignore_existing=True)
            self.yt_locke_client.lock(root)

            data = self.yt_locke_client.get(root)

            if not data:
                data = {}

            for file_name in self.released_bundles_files:
                file_path = os.path.join("/metrika", file_name)
                data[file_path] = formatdate(timeval=self.mtime, localtime=False, usegmt=True)
                logging.info("Set mtime %s for %s in %s", data[file_path], file_path, root)

            self.yt_locke_client.set(root, data)
            self.Context.mtime_json = data
            self.Context.mtime_timestamp_json = {'timestamp': int(time.time())}

    def write_features_json(self):
        root = os.path.join(self.Parameters.yt_locke_root, "misc", "features")

        with self.yt_locke_client.Transaction():
            self.yt_locke_client.create('document', root, recursive=True, ignore_existing=True)
            self.yt_locke_client.lock(root)

            data = self.yt_locke_client.get(root)

            if not data:
                data = {}

            for bundle_name in self.features_mapping:
                data[bundle_name] = self.features_mapping[bundle_name]
                logging.info("Set %s features %s in %s", bundle_name, data[bundle_name], root)

            self.yt_locke_client.set(root, data)
            self.Context.features_json = data

    def create_infra_event(self, api):
        return api.create(self.new_infra_event()).get("id")

    def finish_infra_event(self, api, event_id):
        api.update(event_id, {"finishTime": int(time.time())})

    def new_infra_event(self):
        return {
            "title": "Выкладка бандла {} кода счетчика".format(self.Parameters.bundle_name),
            "description": (
                "Sandbox task: https://sandbox.yandex-team.ru/task/{}/view\n".format(self.id) +
                "Author: https://staff.yandex-team.ru/{}\n".format(self.author) +
                "Bundle: {}\n".format(self.Parameters.bundle_name)
            ),
            "serviceId": 341,  # Metrika Frontend
            "environmentId": 535,  # production
            "startTime": int(time.time()),
            "type": "issue",
            "severity": "major" if "prod" in self.Parameters.bundle_name else "minor",
            "tickets": str(self.Parameters.tracker_issue),
            "meta": {},
            "sendEmailNotifications": True,
            "setAllAvailableDc": True
        }

    def release_sb_bundles(self):
        for bundle_name, bundle in self.bundles_mapping.iteritems():
            logging.debug('Processing %s bunlde', bundle_name)
            bundle_res_path = self.wd_path(bundle_name, 'metrika_watch_bundles')
            bundle_files_path = bundle_res_path.joinpath(bundle_name, 'metrika')
            bundle_files_path.mkdir(parents=True)

            for f_name, save_as in bundle.iteritems():
                shutil.copy(
                    os.path.join(self.watch_data_path, f_name),
                    bundle_files_path.joinpath(save_as).as_posix(),
                )

            bundle_res_path.joinpath('{}_global_mtime.json'.format(bundle_name)).write_text(
                six.text_type(json.dumps(self.Context.mtime_json, sort_keys=True, indent=2)),
                encoding='utf8'
            )
            bundle_res_path.joinpath('{}_global_mtime_timestamp.json'.format(bundle_name)).write_text(
                six.text_type(json.dumps(self.Context.mtime_timestamp_json, sort_keys=True, indent=2)),
                encoding='utf8'
            )
            bundle_res_path.joinpath('{}_watch_version.json'.format(bundle_name)).write_text(
                six.text_type(json.dumps({'version': self.watch_version}, sort_keys=True, indent=2)),
                encoding='utf8'
            )

            MetrikaWatchBundle(
                self,
                'watch deploy {} bundle'.format(bundle_name),
                bundle_res_path,
                bundle=bundle_name,
                watch_version=self.watch_version,
                pack_tar=base.CompressionType.TGZ,
            )
            logging.debug('Successfully processed %s bundle', bundle_name)

    def release_resource(self):
        with sdk2.helpers.ProgressMeter("Releasing bundles files", maxval=len(self.bundles_mapping)) as progress:
            for bundle_name in self.bundles_mapping:
                common_utils.set_resource_attributes(self.Parameters.watch_resource.id, {"bundle_{}_released".format(bundle_name): True, "ttl": "inf"})
                progress.add(1)
