# coding=utf-8
import json
import logging
import os
import shutil

from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.common import binary_task
from sandbox.projects.metrika import utils as metrika_utils
from sandbox.projects.metrika.frontend import utils as frontend_utils
from sandbox.projects.metrika.utils import base_metrika_task, parameters as metrika_parameters
from sandbox.sdk2 import parameters


@base_metrika_task.with_parents
class MetrikaWatchNpmRelease(base_metrika_task.BaseMetrikaTask):
    """
    Загрузка кода счетчика Метрики в NPM
    """

    class Parameters(metrika_utils.CommonParameters):
        description = "Загрузка кода счетчика Метрики в NPM"

        container = metrika_parameters.LastKnightContainerResource("Environment container resource", required=True)

        watch_resource = frontend_utils.LastReleasedWatchJSResource("WatchJS released build")

        npm_version = parameters.String("NPM version (\"npm version minor\" will not run, if version is specified)", default="")

        tracker_issue = metrika_parameters.TrackerIssue(required=True)

        with parameters.Group("Секреты") as secrets_group:
            tokens_secret = parameters.YavSecret("Секрет с токенами", required=True, default="sec-01fs1zy3zbjk0n3hzrnk1p6v3r")

            npm_token_key = parameters.String("Ключ токена NPM в секрете", required=True, default="npm-token")

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def on_execute(self):
        logging.info("\n" + "+" * 80 + " start of on_execute " + "+" * 80)

        from metrika.pylib import sh
        setattr(self, "_sh", sh)
        setattr(self, "_env", frontend_utils.environment_with_node())
        setattr(self, "_abs_path", os.path.abspath("."))
        setattr(self, "_watch_path", os.path.join(self._abs_path, "watchjs"))

        logging.info("Current environment:\n%s" % self._env)

        stages = [
            self.get_watch,
            self.update_package_json,
            self.deploy_to_npm
        ]

        total = len(stages)
        for index, stage in enumerate(stages):
            self.run_stage(stage, index + 1, total)

        logging.info("\n" + "+" * 80 + " end of on_execute " + "+" * 80)

    def get_watch(self):
        logging.info("Getting WatchJS build (package.json + watch.js is used)")
        watch = sdk2.ResourceData(self.Parameters.watch_resource)
        watch_path = "%s/ua" % str(watch.path)
        logging.info("WatchJS build contents: %s" % os.listdir(watch_path))
        os.mkdir(self._watch_path)
        for name in [".npmrc", "package.json", "watch.js", "tag.js"]:
            src = os.path.join(watch_path, name)
            dst = os.path.join(self._watch_path, name)
            logging.info("Copying %s to %s" % (src, dst))
            shutil.copy(src, dst)
            os.chmod(dst, 33260)  # Sandbox sometimes get resource data with 0444 permissions

    def update_package_json(self):
        pkg_json = os.path.join(self._watch_path, "package.json")
        logging.info("Updating %s file" % pkg_json)

        data = open(pkg_json).read()
        logging.info("package.json contents before update:\n%s" % data)

        with open(pkg_json, "r") as fp:
            package = json.load(fp)
            package["version"] = self._get_package_version()
            package["scripts"]["deploy"] = self._get_deploy()

        with open(pkg_json, "w+") as fp:
            json.dump(package, fp, indent=4)

        data = open(pkg_json).read()
        logging.info("package.json contents after update:\n%s" % data)

    def deploy_to_npm(self):
        logging.info("Setting up environment variables")
        self._env["NPM_TOKEN"] = self.Parameters.tokens_secret.data().get(self.Parameters.npm_token_key)

        current = os.getcwd()
        logging.info("Current work dir: %s" % current)

        os.chdir(self._watch_path)
        logging.info("Work dir changed to %s" % self._watch_path)

        logging.info("Deploying to NPM")
        self._run_command("npm run deploy --verbose")

    def _get_package_version(self):
        if self.Parameters.npm_version:
            logging.info("Taking npm_version from task parameters")
            version = self.Parameters.npm_version
        else:
            logging.info("Taking npm_version from NPM")
            version, err = self._run_command("npm view yandex-metrica-watch version")
        version = version.strip()
        logging.info("Package version: %s" % version)
        return version

    def _run_command(self, command, env=None):
        if env is None:
            env = self._env
        logging.info("Running command: %s" % command)
        out, err, retcode = self._sh.run(command, env=env, return_all=True)
        logging.info("Command stdout: %s" % out)
        logging.info("Command stderr: %s" % err)

        if retcode != 0:
            message = "Failed to run command: [%s]. Return code: %s" % (command, retcode)
            raise errors.TaskError(message)
        else:
            return out, err

    def _get_deploy(self):
        logging.info("Generating deploy commands")
        deploy_commands = []
        if not self.Parameters.npm_version:
            logging.info("npm_version does not specified in parameters, adding \"npm version minor\"")
            deploy_commands.append("npm version minor")
        deploy_commands.append("npm publish")
        deploy_commands.append("curl http://purge.jsdelivr.net/npm/yandex-metrica-watch/watch.js")
        deploy_commands.append("curl http://purge.jsdelivr.net/npm/yandex-metrica-watch/tag.js")

        deploy = " && ".join(deploy_commands)
        logging.info("Generated deploy: %s" % deploy)
        return deploy

    @staticmethod
    def run_stage(stage, stage_num, stages_total):
        info = "+  Running stage %s/%s: %s  +" % (stage_num, stages_total, stage.__name__.strip("_"))
        info_len = len(info)
        logging.info("\n" + "+" * info_len + "\n" + info + "\n" + "+" * info_len)
        stage()
