# coding=utf-8
import json
import os

from sandbox import sdk2
from sandbox.common.urls import get_resource_link, get_task_link
from sandbox.projects.common import binary_task, utils as common_utils
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.frontend import metrika_watch_build, metrika_watch_bundle_release
from sandbox.projects.metrika.frontend.metrika_watch_release import release_helper
from sandbox.projects.metrika.utils import base_metrika_task
from sandbox.projects.metrika.utils.parameters import hide
from sandbox.projects.metrika.utils.pipeline import pipeline
from sandbox.sdk2 import parameters


@base_metrika_task.with_parents
class MetrikaWatchRelease(pipeline.PipelineBaseTask):
    """
    Релиз кода счетчика Метрики
    """

    class Context(pipeline.PipelineBaseTask.Context):
        previous_version = None
        version = None
        build_task = []

    class Parameters(utils.CommonParameters):
        description = "Релиз кода счетчика Метрики"

        owner = "METRIKA_BROWSERS"

        is_new = parameters.Bool("Новый релиз", required=True, default=True, description="У нового релиза увеличивается мажорная версия, у старого - минорная.")

        _binary = hide(binary_task.binary_release_parameters_list(stable=True))

    def on_prepare(self):
        self.Context.author = self.real_author

        if not self.Context.previous_version:
            if self.Parameters.is_new:
                self.Context.previous_version = release_helper.ReleaseHelper.get_previous_release_version()
            else:
                self.Context.previous_version = release_helper.ReleaseHelper.get_latest_release_version()
        if not self.Context.version:
            self.Context.version = release_helper.ReleaseHelper.increase_version(self.Context.previous_version, self.Parameters.is_new)

        release_helper.ReleaseHelper.update_description(self, self.Context.author, self.Context.version)

    def create_stages(self):
        return [
            (self.stage_create_release_branch_and_tag, "Релизный бранч и тег"),
            (self.stage_create_release_issue, "Релизный тикет"),
            (self.stage_build, "Сборка"),
            (self.stage_create_bundles_release_tasks, "Релизные таски для бандлов"),
        ]

    def stage_create_release_branch_and_tag(self):
        release_helper.ReleaseHelper.create_release_branch_and_tag(self, self.Context.previous_version, self.Context.version, self.Parameters.is_new)

        release_helper.ReleaseHelper.set_create_release_branch_and_tag_info(self)

    def stage_create_release_issue(self):
        release_helper.ReleaseHelper.create_release_issue(self)

        release_helper.ReleaseHelper.set_create_release_issue_info(self)

    def stage_build(self):
        self.run_subtasks([(
            metrika_watch_build.MetrikaWatchBuild.name,
            {
                "description": "Сборка",
                metrika_watch_build.MetrikaWatchBuild.Parameters.checkout_arcadia_from_url.name:
                    release_helper.ReleaseHelper.get_arcadia_url(release_helper.ReleaseHelper.get_release_tag(self.Context.version)),
                metrika_watch_build.MetrikaWatchBuild.Parameters.env_vars.name:
                    {"DISABLE_DEV_SSH_TUNNEL": "1", "VERSION": release_helper.ReleaseHelper.undot(self.Context.version)},
                metrika_watch_build.MetrikaWatchBuild.Parameters.tracker_issue.name: self.Context.release_issue_key
            }
        )], subtasks_variable=self.Context.build_task, after_subtasks_waited=self.__set_resource_version_attributes)

    def stage_create_bundles_release_tasks(self):
        tracker_issue = self.Context.release_issue_key
        resource = release_helper.ReleaseHelper.get_watch_resource(self)
        comments = []

        resource_link = "(({} {} #{}))".format(get_resource_link(resource.id), resource.type.name, resource.id)
        comments.append("**!!(green)версия {}!!**".format(self.Context.version))
        comments.append("ресурс **{}**".format(resource_link))
        comments.append("\n")
        comments.append("#|")
        comments.append("|| бандл | задача для выкладки ||")

        comments.extend(self.create_bundles_release_task(["preprod", "euro-preprod"], resource, tracker_issue))
        comments.extend(self.create_bundles_release_task(["prod", "euro", "yandex", "yandex-team", "new-counter-experiment"], resource, tracker_issue))
        comments.extend(self.create_bundles_release_task(["assessor-experiment"], resource, tracker_issue))

        comments.append("|#\n")
        comments = comments + self.__create_acceptance_comment()
        self.st_client.issues[tracker_issue].comments.create(text="\n".join(comments))

    def create_bundles_release_task(self, bundle_names, resource, tracker_issue):
        task = metrika_watch_bundle_release.MetrikaWatchBundleRelease(
            self,
            description=self.Parameters.description + " (бандлы {})".format(", ".join(bundle_names)),
            owner=self.Parameters.owner,
            bundle_names=bundle_names,
            watch_resource=resource,
            tracker_issue=tracker_issue,
            release_npm=("prod" in bundle_names),
        )
        task.save()

        comments = []
        task_link = "(({} {} #{}))".format(get_task_link(task.id), task.type.name, task.id)
        for bundle_name in bundle_names:
            comment = "|| **{}** | **{}** ||".format(
                bundle_name,
                task_link
            )
            comments.append(comment)

        return comments

    # создаем ссылки для приемки
    def __create_acceptance_comment(self):
        resource_prod_data = release_helper.ReleaseHelper.get_release_prod_resource_data()
        meta_data_prod = json.load(open(os.path.join(resource_prod_data.path.as_posix(), 'meta.json')))
        resource_new = release_helper.ReleaseHelper.get_watch_resource(self)
        meta_data_new = json.load(open(os.path.join(sdk2.ResourceData(resource_new).path.as_posix(), 'meta.json')))
        link_base = "https://analytics-metrika.in.yandex.net/code_versions_2?"
        pairs = release_helper.ReleaseHelper.get_pairs_to_compare(
            meta_data_prod,
            meta_data_new,
        )
        comments = [
            "<{ссылки для генерации приемки\n",
            "!!проверь версии, при ручных выкладках автоматика могла дать сбой!!\n",
            "**версии !!(blue){}!! vs !!(green){}!!**\n".format(
                pairs[0].get('current').get('version'),
                pairs[0].get('new').get('version'),
            ),
            "#|",
            "|| файлы | бандлы | | ссылка ||",
        ]

        for pair in pairs:
            current = pair.get('current')
            new = pair.get('new')
            query_params = "&".join([
                "code_version_prod={}".format(current.get('version')),
                "code_version_preprod={}".format(new.get('version')),
                "code_features_prod={}".format(','.join(current.get('features'))),
                "code_features_preprod={}".format(','.join(new.get('features'))),
                "type_of={}".format(pair.get('type'))
            ])
            table_row = "|| {} | {} | {} | {} ||".format(
                "**{}**".format(', '.join(pair.get('files'))),
                "!!(blue){}!!".format(", ".join(current.get('bundles'))),
                "!!(green){}!!".format(", ".join(new.get('bundles'))),
                "(({} ссылка))".format(link_base + query_params)
            )
            comments.append(table_row)

        comments.append("|#")
        comments.append("}>")
        return comments

    def __set_resource_version_attributes(self):
        resource = release_helper.ReleaseHelper.get_watch_resource(self)
        common_utils.set_resource_attributes(
            resource.id,
            {
                "watch_version": self.Context.version,
                "watch_undotted_version": release_helper.ReleaseHelper.undot(self.Context.version),
            }
        )
