# coding=utf-8
import logging
import re
from distutils import version

from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.common.vcs import arc
from sandbox.projects.metrika.utils import resource_types, vcs
from sandbox.projects.metrika.java.utils import metrika_java_helper
from sandbox.projects.metrika.utils import settings
from sandbox.projects.metrika.utils import arcanum_api
from sandbox.sdk2.vcs import svn

logger = logging.getLogger("release-helper")

pairs_to_compare = [
    {
        'type': 'watch',
        'current_bundles': [
            'prod',
            'new-counter-experiment',
        ],
        'new_bundles': [
            'preprod',
        ],
        'files': [
            'watch.js',
        ],
    },
    {
        'type': 'watch',
        'current_bundles': [
            'prod',
            'new-counter-experiment',
        ],
        'new_bundles': [
            'preprod',
        ],
        'files': [
            'watch_serp.js',
        ],
    },
    {
        'type': 'new_counter',
        'current_bundles': [
            'prod',
            'new-counter-experiment',
        ],
        'new_bundles': [
            'preprod',
        ],
        'files': [
            'tag_turbo.js',
        ],
    },
    {
        'type': 'tag',
        'current_bundles': [
            'prod',
            'new-counter-experiment',
        ],
        'new_bundles': [
            'preprod',
        ],
        'files': [
            'tag.js',
        ],
    },
    {
        'type': 'tag',
        'current_bundles': [
            'prod',
            'new-counter-experiment',
        ],
        'new_bundles': [
            'preprod',
        ],
        'files': [
            'tag_turboapp.js',
        ],
    },
    {
        'type': 'new_counter',
        'current_bundles': [
            'euro',
        ],
        'new_bundles': [
            'euro-preprod',
        ],
        'files': [
            'tag.js',
            'watch.js',
        ],
    },
]


def get_pair_to_compare(config_item, key, release_meta):
    version = ''
    features = []
    config_item_bundles = config_item.get(key)
    config_item_files = config_item.get('files')

    for bundle in release_meta:
        if bundle.get('bundle') in config_item_bundles:
            for file in bundle.get('files'):
                if file.get('file') in config_item_files:
                    version = file.get('version')
                    features.append(file.get('features'))

    return {
        'version': version,
        'features': features,
        'bundles': config_item_bundles,
    }


def get_pairs_to_compare(config, current_release_meta, new_release_meta):
    result = []
    for config_item in config:
        result.append({
            'current': get_pair_to_compare(config_item, 'current_bundles', current_release_meta),
            'new': get_pair_to_compare(config_item, 'new_bundles', new_release_meta),
            'type': config_item.get('type'),
            'files': config_item.get('files'),
        })

    return result


class ReleaseHelper(object):
    release_branch_prefix = "releases/metrika/watch/"
    release_tag_prefix = "tags/releases/metrika/watch/"

    @staticmethod
    def get_pairs_to_compare(current_release_meta, new_release_meta):
        return get_pairs_to_compare(pairs_to_compare, current_release_meta, new_release_meta)

    @staticmethod
    def get_release_prod_resource_data():
        resource = sdk2.Resource.find(
            type=resource_types.MetrikaWatch,
            attrs={"bundle_prod_released": True},
            limit=1
        ).first()
        return sdk2.ResourceData(resource)

    @staticmethod
    def get_previous_release_version():
        last_released_watch_resource = sdk2.Task.server.resource.read(
            type=resource_types.MetrikaWatch,
            attrs={"bundle_prod_released": True},
            limit=1
        ).get("items")[0]
        build_task = sdk2.Task[last_released_watch_resource.get("task").get("id")]
        previous_release_tag = svn.Arcadia.get_revision(build_task.Parameters.checkout_arcadia_from_url)
        return ReleaseHelper.get_version_from_tag(previous_release_tag)

    @staticmethod
    def get_latest_release_version():
        arcanum = arcanum_api.ArcanumApi(token=sdk2.yav.Secret(settings.yav_uuid).data()["arcanum-token"])
        all_branches = arcanum.get_branches(ReleaseHelper.release_tag_prefix)
        return str(max(
            [version.LooseVersion(ReleaseHelper.get_version_from_tag(branch)) for branch in all_branches if branch.startswith(ReleaseHelper.release_tag_prefix)]
        ))

    @staticmethod
    def get_version_from_tag(tag):
        return tag.replace(ReleaseHelper.release_tag_prefix, "")

    @staticmethod
    def increase_version(release_version, is_new):
        major, minor = version.LooseVersion(release_version).version
        if is_new:
            major += 1
            minor = 0
        else:
            minor += 1
            if minor > 9:
                raise errors.TaskError("Минорная версия не может быть больше 9! Создайте новый релиз.")
        return ReleaseHelper.build_version(major, minor)

    @staticmethod
    def build_version(major, minor):
        return "{}.{}".format(major, minor)

    @staticmethod
    def update_description(task, author, version):
        metrika_java_helper.MetrikaJavaHelper._update_description(task, "версии {} от {}".format(
            metrika_java_helper.MetrikaJavaHelper._grey(version),
            metrika_java_helper.MetrikaJavaHelper._blue(author)
        ))

    @staticmethod
    def get_arcadia_url(revision):
        return "{}:/#{}".format(svn.Arcadia.ARCADIA_ARC_SCHEME, revision)

    @staticmethod
    def create_release_branch_and_tag(task, previous_release_version, release_version, is_new):
        arc_client = arc.Arc()
        release_branch = ReleaseHelper.get_release_branch(release_version)
        start_point_url = ReleaseHelper.get_arcadia_url("trunk" if is_new else release_branch)
        with vcs.mount_arc(start_point_url) as arcadia:
            if is_new:
                arc_client.checkout(arcadia, release_branch, True)
                arc_client.push(arcadia, release_branch)
            arc_client.push(arcadia, refspecs=[("HEAD", ReleaseHelper.get_release_tag(release_version))])

            start = ReleaseHelper.get_release_tag(previous_release_version)
            end = ReleaseHelper.get_release_tag(release_version)
            with vcs.mount_arc(ReleaseHelper.get_arcadia_url("trunk")) as arcadia:
                arc_client.fetch(arcadia, start)
                arc_client.fetch(arcadia, end)
                task.Context.changelog = arc_client.log(arcadia, path="metrika/frontend/watch", start_commit=start, end_commit=end)
                task.Context.short_log = list(set(re.findall(r".*?([A-Z]+-\d+).*?", task.Context.changelog)))

    @staticmethod
    def get_release_branch(release_version):
        return ReleaseHelper.release_branch_prefix + ReleaseHelper.get_release_major(release_version)

    @staticmethod
    def get_release_tag(release_version):
        return ReleaseHelper.release_tag_prefix + release_version

    @staticmethod
    def create_release_issue(task):
        search_parameters = {
            "queue": "METR",
            "components": 1453,
            "tags": [ReleaseHelper.get_release_branch(task.Context.version)],
            "type": "release",
        }
        create_parameters = search_parameters.copy()
        create_parameters.update({
            "summary": "Релиз кода счетчика {}".format(ReleaseHelper.get_release_major(task.Context.version)),
            "assignee": task.Context.author,
            "followers": ["nkatherine", "eugenenik", "sergeydenisov", "miptgirl", "stanislavsky"]
        })

        issue = metrika_java_helper.MetrikaJavaHelper.find_or_create_issue(task.st_client, search_parameters, create_parameters, task.id, sprint_board=9501)
        logger.info("Issue {} created.".format(issue.key))
        task.Context.release_issue_key = issue.key

        comment = "**!!(green)Версия {}!!**\n\n".format(task.Context.version)
        comment += "<{{**Тикеты**\n{}\n}}>\n".format("\n".join(task.Context.short_log) if task.Context.short_log else "Отсутствуют")
        comment += "<{{**Полный лог**\n%%\n{}\n%%\n}}>".format(task.Context.changelog or "Не содержит коммитов")

        issue.comments.create(text=comment)
        return issue.key

    @staticmethod
    def get_release_major(release_version):
        major, minor = version.LooseVersion(release_version).version
        return str(major)

    @staticmethod
    def undot(release_version):
        return release_version.replace(".", "")

    @staticmethod
    def get_watch_resource(task):
        return sdk2.Resource.find(task_id=task.Context.build_task[0], type=resource_types.MetrikaWatch).first()

    @staticmethod
    def set_create_release_branch_and_tag_info(task):
        template = "Ветка и теги: <a href=\"https://a.yandex-team.ru/arc_vcs/branches/?filterName={0}\">{0}</a>"
        create_release_branch_and_tag_info = template.format(ReleaseHelper.get_release_branch(task.Context.version))

        task.set_info(create_release_branch_and_tag_info, do_escape=False)

    @staticmethod
    def set_create_release_issue_info(task):
        create_release_issue_info = "Задача: <a href=\"https://st.yandex-team.ru/{0}\">{0}</a>".format(task.Context.release_issue_key)

        task.set_info(create_release_issue_info, do_escape=False)
