# coding=utf-8
import json
import logging
import os

import requests
from six.moves.urllib import parse

from sandbox.projects.common.decorators import retries
from sandbox.projects.metrika.utils.mixins.console import BaseConsoleMixin
from sandbox.projects.metrika.utils.resource_types import LastReleasedWatchJSResource  # noqa
from sandbox.sdk2 import helpers


def environment_with_node():
    environment = os.environ.copy()
    environment["PATH"] = "/opt/nodejs/8/bin:" + environment["PATH"]
    return environment


def zopfli_and_brotli(path, ignore_extensions=None):
    if ignore_extensions:
        filter = ' '.join('-not -iname "*.{}"'.format(extension) for extension in ignore_extensions)
    else:
        filter = ''
    helpers.subprocess.check_call(
        "find {path} -type f {filter} | xargs -I {{}} -P 0 zopfli {{}}".format(path=path, filter=filter),
        stderr=helpers.subprocess.STDOUT, shell=True
    )
    helpers.subprocess.check_call(
        'find {path} -type f -not -iname "*.gz" {filter} | xargs -I {{}} -P 0 brotli --quality 11 --input {{}} --output {{}}.br --force'.format(path=path, filter=filter),
        stderr=helpers.subprocess.STDOUT, shell=True
    )


class FnmMixin(BaseConsoleMixin):
    @retries(3)
    def _download(self):
        self._execute_script('curl -fsSL https://github.com/Schniz/fnm/raw/v1.27.0/.ci/install.sh | bash -s -- -r v1.27.0')

    def install_node(self, version):
        self._download()
        with helpers.ProcessLog(self, logger='console', stderr_level=logging.ERROR) as pl:
            os.environ['PATH'] = helpers.subprocess.check_output(
                'export PATH=$HOME/.fnm:$PATH && eval "`fnm env`" '
                '&& fnm install {0} 1>&2 && fnm use {0} 1>&2 && echo $PATH'.format(version),
                shell=True, stderr=pl.stderr
            ).strip()


DOCKER_INFO_URL = "https://dockinfo.yandex-team.ru/api/docker/"
REGISTRY_URL = "registry.yandex.net"


class Docker:

    @staticmethod
    def get_full_image_name(name):
        return REGISTRY_URL + "/" + name

    @staticmethod
    def get_full_image_url(name, tag):
        return Docker.get_full_image_name(name) + ":" + tag

    def __init__(self, token):
        self.__token = token

    def get_image_hash(self, name, tag="latest"):
        return self.__docker_info_request("hash", {"registryUrl": self.get_full_image_name(name), "tag": tag}).text

    def get_latest_tag(self, name):
        return json.loads(self.__docker_info_request("suggest", {"pattern": self.get_full_image_name(name)}).text)[0]["tag"]

    def __docker_info_request(self, path, params):
        response = requests.get(parse.urljoin(DOCKER_INFO_URL, path), params=params, headers={"Authorization": "OAuth {}".format(self.__token)}, verify=False)
        assert response.status_code == 200
        return response
